# LogisInvWeibull.R
#' Logistic Inverse Weibull Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic Inverse Weibull distribution.
#'
#' The Logistic Inverse Weibull distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic Inverse Weibull distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1}{1+\left(e^{\lambda x^{-\beta}}-1\right)^\alpha} \, ; \, x > 0.
#' }
#'  
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.inv.weib()} — Density function
#'   \item \code{plogis.inv.weib()} — Distribution function
#'   \item \code{qlogis.inv.weib()} — Quantile function
#'   \item \code{rlogis.inv.weib()} — Random generation
#'   \item \code{hlogis.inv.weib()} — Hazard function
#' }
#'
#' @usage
#' dlogis.inv.weib(x, alpha, beta, lambda, log = FALSE)
#' plogis.inv.weib(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.inv.weib(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.inv.weib(n, alpha, beta, lambda)
#' hlogis.inv.weib(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.inv.weib}: numeric vector of (log-)densities
#'   \item \code{plogis.inv.weib}: numeric vector of probabilities
#'   \item \code{qlogis.inv.weib}: numeric vector of quantiles
#'   \item \code{rlogis.inv.weib}: numeric vector of random variates
#'   \item \code{hlogis.inv.weib}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary,A.K., & Kumar, V.(2020). 
#' A Study on Properties and Goodness-of-Fit of The Logistic Inverse Weibull 
#' Distribution. \emph{Global Journal of Pure and Applied Mathematics(GJPAM)},
#' \bold{16(6)},871--889. 
#' \doi{10.37622/GJPAM/16.6.2020.871-889}   
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dlogis.inv.weib(x, 2.5, 1.5, 0.1)
#' plogis.inv.weib(x, 2.5, 1.5, 0.1)
#' qlogis.inv.weib(0.5, 2.5, 1.5, 0.1)
#' rlogis.inv.weib(10, 2.5, 1.5, 0.1)
#' hlogis.inv.weib(x, 2.5, 1.5, 0.1)
#' 
#' # Data
#' x <- stress31
#' # ML estimates
#' params = list(alpha=22.20247, beta=0.34507, lambda=3.74216)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.inv.weib, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.inv.weib, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.inv.weib, pfun=plogis.inv.weib, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisInvWeibull
#' @aliases dlogis.inv.weib plogis.inv.weib qlogis.inv.weib rlogis.inv.weib hlogis.inv.weib
#' @export
NULL

#' @rdname LogisInvWeibull
#' @usage NULL
#' @export   
dlogis.inv.weib <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	z <- exp(lambda *(x[valid] ^ -beta))
    num <- alpha * beta *lambda *(x[valid] ^-(beta+1.0)) * z * ((z-1)^(alpha - 1.0))
    deno <- (1.0 + (z-1) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisInvWeibull
#' @usage NULL
#' @export   
plogis.inv.weib <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(lambda *(q[valid] ^ -beta))
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0/tmp
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisInvWeibull
#' @usage NULL
#' @export 
qlogis.inv.weib <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	qtl[valid] <- ((1/lambda) * log(1 + (((1 - p[valid])/p[valid]) ^ (1/alpha)))) ^ (-1/beta)  
	}
    return(qtl)   
}

#' @rdname LogisInvWeibull
#' @usage NULL
#' @export  
rlogis.inv.weib <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.inv.weib(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisInvWeibull
#' @usage NULL
#' @export 
hlogis.inv.weib <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
     
    nume <- dlogis.inv.weib(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.inv.weib(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}    


