% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obn_fast_mdlp.R
\name{ob_numerical_fast_mdlp}
\alias{ob_numerical_fast_mdlp}
\title{Optimal Binning using MDLP with Monotonicity Constraints}
\usage{
ob_numerical_fast_mdlp(
  feature,
  target,
  min_bins = 2L,
  max_bins = 5L,
  bin_cutoff = 0.05,
  max_n_prebins = 100L,
  convergence_threshold = 1e-06,
  max_iterations = 1000L,
  force_monotonicity = TRUE
)
}
\arguments{
\item{feature}{A numeric vector representing the continuous predictor variable.
Missing values (NA) are excluded during the binning process.}

\item{target}{An integer vector of binary outcomes (0/1) corresponding to
each observation in \code{feature}. Must have the same length as \code{feature}.}

\item{min_bins}{Integer. The minimum number of bins to produce. Must be \eqn{\ge} 2.
Defaults to 2.}

\item{max_bins}{Integer. The maximum number of bins to produce. Must be \eqn{\ge}
\code{min_bins}. Defaults to 5.}

\item{bin_cutoff}{Numeric. Currently unused in this implementation (reserved for
future versions). Defaults to 0.05.}

\item{max_n_prebins}{Integer. Currently unused in this implementation (reserved for
future versions). Defaults to 100.}

\item{convergence_threshold}{Numeric. The threshold for determining convergence
during the iterative monotonicity enforcement process. Defaults to 1e-6.}

\item{max_iterations}{Integer. Safety limit for the maximum number of iterations
in the monotonicity enforcement phase. Defaults to 1000.}

\item{force_monotonicity}{Logical. If \code{TRUE}, the algorithm enforces a strict
monotonic relationship (increasing or decreasing) between the bin indices
and their Weight of Evidence (WoE) values. Defaults to \code{TRUE}.}
}
\value{
A list containing the binning results:
  \itemize{
    \item \code{id}: Integer vector of bin identifiers.
    \item \code{bin}: Character vector of bin labels in interval notation.
    \item \code{woe}: Numeric vector of Weight of Evidence for each bin.
    \item \code{iv}: Numeric vector of Information Value contribution per bin.
    \item \code{count}: Integer vector of total observations per bin.
    \item \code{count_pos}: Integer vector of positive cases.
    \item \code{count_neg}: Integer vector of negative cases.
    \item \code{cutpoints}: Numeric vector of upper boundaries (excluding Inf).
    \item \code{converged}: Logical indicating if the monotonicity enforcement converged.
    \item \code{iterations}: Integer count of iterations in monotonicity phase.
  }
}
\description{
Performs supervised discretization of continuous numerical variables using the
Minimum Description Length Principle (MDLP) algorithm, enhanced with optional
monotonicity constraints on the Weight of Evidence (WoE). This method is
particularly suitable for creating interpretable bins for logistic regression
models in domains like credit scoring.
}
\details{
This function implements a sophisticated hybrid approach combining the classic
MDLP algorithm with modern monotonicity constraints.

\strong{Algorithm Pipeline:}
\enumerate{
  \item \strong{Data Preparation:} Removes NA values and sorts the data by feature value.
  \item \strong{MDLP Discretization (Fayyad & Irani, 1993):}
  \itemize{
    \item Recursively evaluates all possible binary splits of the sorted data.
    \item For each potential split, calculates the Information Gain (IG).
    \item Applies the MDLP stopping criterion:
          \deqn{IG > \frac{\log_2(N-1) + \Delta}{N}}
          where \eqn{N} is the total number of samples and \eqn{\Delta = \log_2(3^k - 2) - k \cdot E(S)}
          (for binary classification, \eqn{k=2}).
    \item Only accepts splits that significantly reduce entropy beyond what would
          be expected by chance, balancing model fit with complexity.
  }
  \item \strong{Constraint Enforcement:}
  \itemize{
    \item \strong{Min/Max Bins:} Adjusts the number of bins to meet \code{[min_bins, max_bins]}
          requirements through intelligent splitting or merging.
    \item \strong{Monotonicity (if enabled):} Iteratively merges adjacent bins with
          the most similar WoE values until a strictly increasing or decreasing
          trend is achieved across all bins.
  }
}

\strong{Technical Notes:}
\itemize{
  \item The algorithm uses Laplace smoothing (\eqn{\alpha = 0.5}) when calculating
        WoE to prevent \eqn{\log(0)} errors for bins with pure class distributions.
  \item When all feature values are identical, the algorithm creates artificial bins.
  \item The monotonicity enforcement phase is iterative and uses the
        \code{convergence_threshold} to determine when changes in WoE become negligible.
}
}
\examples{
# Example: Standard usage with monotonicity
set.seed(123)
feature <- rnorm(1000)
target <- rbinom(1000, 1, plogis(2 * feature)) # Positive relationship

result <- ob_numerical_fast_mdlp(feature, target,
  min_bins = 3,
  max_bins = 6,
  force_monotonicity = TRUE
)

print(result$bin)
print(result$woe) # Should show a monotonic trend

# Example: Disabling monotonicity for exploratory analysis
result_no_mono <- ob_numerical_fast_mdlp(feature, target,
  min_bins = 3,
  max_bins = 6,
  force_monotonicity = FALSE
)

print(result_no_mono$woe) # May show non-monotonic patterns

}
\references{
Fayyad, U. M., & Irani, K. B. (1993). Multi-interval discretization of continuous-valued
attributes for classification learning. \emph{Proceedings of the 13th International
Joint Conference on Artificial Intelligence}, 1022-1029.

Kurgan, L. A., & Musilek, P. (2006). A survey of techniques. \emph{IEEE Transactions
on Knowledge and Data Engineering}, 18(5), 673-689.

Garcia, S., Luengo, J., & Herrera, F. (2013). Data preprocessing in data mining.
\emph{Springer Science & Business Media}.
}
\seealso{
\code{\link{ob_numerical_cm}} for ChiMerge-based approaches,
\code{\link{ob_numerical_dp}} for dynamic programming methods.
}
