#'@title Import ENphylo_modeling results into global environment
#'@description The function retrieves the ENFA/ENphylo models generated by
#'  \code{\link{ENphylo_modeling}} and arranges them into a named list to be
#'  used in \code{\link{ENphylo_prediction}} function. It also offers the option
#'  to choose between retrieving ENFA or ENphylo models and to select the model
#'  produced for one or more focal species.
#'@usage
#'getENphylo_results(input.dir,mods="all",species_name=NULL,only_evaluations=FALSE)
#'@param input.dir the file path wherein the folders "ENphylo_enfa_models" and
#'  "ENphylo_imputed_models" generated by \code{\link{ENphylo_modeling}}
#'  are stored.
#'@param mods character. Name of the models to be retrieved. Viable options are:
#'  \code{“enfa”} (enfa models), \code{“enphylo”} (ENphylo models), \code{“all”}
#'  (default, enfa plus ENphylo models).
#'@param species_name character. The name of the single (or multiple) species
#'  for which model results must be imported.
#'@param only_evaluations logical. If TRUE, \code{getENphylo_results} returnes
#'  model performances only.
#'@author Alessandro Mondanaro, Silvia Castiglione, Pasquale Raia
#'@return A named list of outputs as described in
#'  \code{\link{ENphylo_modeling}}.
#'@export
#'@seealso \link{ENphylo_modeling}; \href{../doc/ENphylo.html}{\code{getENphylo_results} vignette}
#'@examples
#'\dontrun{
#' library(ape)
#' library(terra)
#' library(sf)
#' library(RRgeo)
#'
#' newwd<-tempdir()
#' # newwd<-"YOUR_DIRECTORY"
#' latesturl<-RRgeo:::get_latest_version("12734585")
#' curl::curl_download(url = paste0(latesturl,"/files/dat.Rda?download=1"),
#'                     destfile = file.path(newwd,"dat.Rda"), quiet = FALSE)
#' load(file.path(newwd,"dat.Rda"))
#' read.tree(system.file("exdata/Eucopdata_tree.txt", package="RRgeo"))->tree
#' tree$tip.label<-gsub("_"," ",tree$tip.label)
#' curl::curl_download(paste0(latesturl,"/files/X35kya.tif?download=1"),
#'                     destfile = file.path(newwd,"X35kya.tif"), quiet = FALSE)
#' rast(file.path(newwd,"X35kya.tif"))->map35
#' project(map35,st_crs(dat[[1]])$proj4string,res = 50000)->map
#'
#' ENphylo_modeling(input_data=dat[c(1,11)],
#'                  tree=tree,
#'                  input_mask=map[[1]],
#'                  obs_col="OBS",
#'                  time_col="age",
#'                  min_occ_enfa=15,
#'                  boot_test_perc=20,
#'                  boot_reps=10,
#'                  swap.args=list(nsim=5,si=0.2,si2=0.2),
#'                  eval.args=list(eval_metric_for_imputation="AUC",
#'                                 eval_threshold=0.7,
#'                                 output_options="best"),
#'                  clust=NULL,
#'                  output.dir=newwd)
#'
#' getENphylo_results(input.dir =newwd,
#'                    mods="all",
#'                    species_name=names(dat)[c(1,11)])->mod
#'
#'}

getENphylo_results<-function (input.dir,
                              mods = "all",
                              species_name=NULL,
                              only_evaluations = FALSE)
{
  if (mods == "enfa") {
    all <- list.files(file.path(path = input.dir, "ENphylo_enfa_models"),
                      full.names = T, recursive = T)
    if (length(all) < 1) {
      stop("No ENFA models within the folder directory")
    }
  }
  if (mods == "enphylo") {
    all <- list.files(file.path(path = input.dir, "ENphylo_imputed_models"),
                      full.names = T, recursive = T)
    if (length(all) < 1) {
      stop("No ENphylo models within the folder directory")
    }
  }
  if (mods == "all") {
    a1 <- list.files(file.path(path = input.dir, "ENphylo_enfa_models"),
                     full.names = T, recursive = T)
    a2 <- list.files(file.path(path = input.dir, "ENphylo_imputed_models"),
                     full.names = T, recursive = T)
    all <- c(a1, a2)
    if (length(a1) < 1 & length(a2) >= 1) {
      warning("No ENFA models within the folder directory")
    }
    if (length(a1) >= 1 & length(a2) < 1) {
      warning("No ENphylo models within the folder directory")
    }
    if (length(all) < 1) {
      stop("No models within the folder directory")
    }
  }

  if (!is.null(species_name)){
    lapply(species_name,function(j){
      all<- all[grep(paste0(j,"/model.output"),all)]
    })->all

  } else {
    all <- all[grep("model_outputs", all)]}

  all_data <- lapply(all, function(x) {
    load(x)
    model_outputs
  })
  all_data <- unlist(all_data, recursive = FALSE)
  if (only_evaluations) {
    all_data <- lapply(all_data, function(x) {
      data.frame(x$calibrated_model$evaluation)
    })
  }
  return(all_data)
}
