% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aoe_border.R
\name{aoe_border}
\alias{aoe_border}
\title{Classify Points by Distance from a Border}
\usage{
aoe_border(
  points,
  border,
  width = NULL,
  area = NULL,
  halo_width = NULL,
  halo_area = NULL,
  mask = NULL,
  bbox = NULL,
  side_names = c("side_1", "side_2"),
  coords = NULL
)
}
\arguments{
\item{points}{An \code{sf} object with POINT geometries, or a data.frame with
coordinate columns.}

\item{border}{An \code{sf} object with LINESTRING or MULTILINESTRING geometry
representing the border.}

\item{width}{Buffer width in meters (for projected CRS) or degrees (for
geographic CRS). Creates core zone within this distance of the border.
Cannot be used together with \code{area}.}

\item{area}{Target area for each side's core zone. The function finds the
buffer width that produces this area per side. If \code{mask} is provided,
the width is adjusted to achieve the target area \emph{after} masking.
Cannot be used together with \code{width}.}

\item{halo_width}{Width of the halo zone beyond the core. If \code{NULL}
(default), equals the core width for symmetric zones.}

\item{halo_area}{Target area for each side's halo zone. Alternative to
\code{halo_width}. If \code{NULL} and \code{halo_width} is \code{NULL}, defaults to equal
area as core.}

\item{mask}{Optional mask for clipping the buffer zones. Can be:
\itemize{
\item \code{sf} object with POLYGON or MULTIPOLYGON geometry
\item \code{"land"}: use the bundled global land mask to exclude sea areas
}}

\item{bbox}{Optional bounding box to limit the study area. Can be:
\itemize{
\item \code{sf} or \code{sfc} object (uses its bounding box)
\item Named vector: \code{c(xmin = ..., ymin = ..., xmax = ..., ymax = ...)}
\item \code{NULL}: no bbox restriction (uses buffer extent)
}}

\item{side_names}{Character vector of length 2 naming the sides.
Default is \code{c("side_1", "side_2")}. The first name is assigned to
the left side of the border (when traversing from start to end).}

\item{coords}{Column names for coordinates when \code{points} is a data.frame.}
}
\value{
An \code{aoe_border_result} object (extends \code{sf}) containing classified
points with columns:
\describe{
\item{point_id}{Original point identifier}
\item{side}{Which side of the border: value from \code{side_names}}
\item{aoe_class}{Distance class: \code{"core"} or \code{"halo"}}
}
Points outside the study area are pruned (not returned).
}
\description{
Given a set of points and a border (line), \code{aoe_border()} classifies
points by their side relative to the border and their distance from it.
Creates equal-area buffer zones on both sides of the border.
}
\details{
The function creates symmetric buffer zones around a border line:
\enumerate{
\item \strong{Core zone}: Points within \code{width} (or \code{area}) distance of the border
\item \strong{Halo zone}: Points beyond core but within \code{width + halo_width}
\item \strong{Pruned}: Points outside the halo zone (not returned)
}

Each zone is split by the border line to determine which side the point
falls on.
\subsection{Equal area across sides}{

When using the \code{area} parameter, the buffer width is calculated to produce
equal area on both sides of the border. With masking, the width is adjusted
so that the \emph{masked} area on each side equals the target.
}
}
\examples{
library(sf)

# Create a border line
border <- st_as_sf(
  data.frame(id = 1),
  geometry = st_sfc(st_linestring(matrix(
    c(0, 0, 100, 100), ncol = 2, byrow = TRUE
  ))),
  crs = 32631
)

# Create points
pts <- st_as_sf(
  data.frame(id = 1:6),
  geometry = st_sfc(
    st_point(c(10, 20)),   # near border, side 1
    st_point(c(30, 10)),   # near border, side 2
    st_point(c(50, 80)),   # far from border, side 1
    st_point(c(80, 40)),   # far from border, side 2
    st_point(c(5, 5)),     # very close to border
    st_point(c(200, 200))  # outside study area
  ),
  crs = 32631
)

# Classify by distance from border
result <- aoe_border(pts, border, width = 20)

}
