% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/placebo.R
\name{placebo_temporal}
\alias{placebo_temporal}
\title{Temporal Placebo Test via Time-Index Permutations}
\usage{
placebo_temporal(
  DT,
  spec = "C",
  k = 2,
  controls = character(0),
  n_perm = 10,
  seed = 999,
  dir_csv = NULL
)
}
\arguments{
\item{DT}{A \code{data.table} (or \code{data.frame}) containing the data
used by \code{fit_one()}.}

\item{spec}{Character scalar; model specification (e.g. \code{"A"},
\code{"B"}, \code{"C"}, \code{"D"}) passed to \code{fit_one()}.}

\item{k}{Integer; lag order passed to \code{fit_one()}.}

\item{controls}{Character vector of control variable names passed to
\code{fit_one()}.}

\item{n_perm}{Integer; number of temporal permutations (placebo
datasets) to run.}

\item{seed}{Integer; base random seed used for reproducibility of the
original fit and the permutations.}

\item{dir_csv}{Character scalar; directory path to save the summary CSV.
If \code{NULL} (default), the CSV is not saved to disk.}
}
\value{
A \code{data.frame} with one row per permutation and columns:
\itemize{
\item \code{perm}: permutation index (1, \dots, \code{n_perm}).
\item \code{elpd_orig}: ELPD of the original (non-permuted) fit.
\item \code{elpd_perm}: ELPD of the model fit on the permuted data.
\item \code{diff}: difference \code{elpd_orig - elpd_perm}.
}
}
\description{
Implements a temporal placebo test for the bivariate hurdle model by
randomly permuting the time ordering of \code{DT}, re-estimating the
model on each permuted dataset, and comparing the PSIS-LOO ELPD of the
original fit against the permuted fits.
}
\details{
The function:
\itemize{
\item Fits the model on the original \code{DT} via \code{fit_one()},
extracts \code{"log_lik_joint"} and computes PSIS-LOO (with
\code{moment_match = TRUE}).
\item For each of \code{n_perm} iterations, permutes the row order of
\code{DT}, refits the model on the permuted data, recomputes
PSIS-LOO, and stores the permuted ELPD.
\item Reports, for each permutation, the original ELPD, the permuted
ELPD, and their difference (\code{elpd_orig - elpd_perm}).
}

This procedure evaluates whether the temporal structure captured by the
model is informative: if the model is exploiting genuine time dependence,
the original ELPD should typically be higher than that of the permuted
(time-scrambled) datasets.

The function assumes that \code{fit_one()} is available in the search path.
}
\examples{
\donttest{
# 1. Create a temporary directory for output
tmp_dir <- file.path(tempdir(), "placebo_out")
dir.create(tmp_dir, showWarnings = FALSE)

# 2. Create dummy data (DT)
# Needed because R CMD check runs in a clean environment
N <- 50
DT <- data.frame(
  time = 1:N,
  y = rpois(N, lambda = 4),
  X1 = rnorm(N),
  X2 = rnorm(N)
)
# Ensure it's a data.table if fit_one expects it, or leave as DF
# (The function internally ensures data.table behavior)

# 3. Define auxiliary parameters
k_grid  <- 0:1

# 4. Run the function
# We use a small n_perm for the example to run faster
try({
  out_placebo <- placebo_temporal(DT, spec = "C", k = 1,
                                  controls = c("X1", "X2"),
                                  n_perm = 2, seed = 999,
                                  dir_csv = tmp_dir)
  
  head(out_placebo)
})

# 5. Cleanup
unlink(tmp_dir, recursive = TRUE)
}

}
