% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bnpMTPfunction.R
\name{bnpMTP}
\alias{bnpMTP}
\title{Bayesian Nonparametric Sensitivity Analyses of Multiple Testing Procedures for p-values with Arbitrary Dependence (Intercorrelations)}
\usage{
bnpMTP( p = NULL , alpha = 0.05 , N = 1000 , mu = 1 )
}
\arguments{
\item{p}{A vector of \emph{p}-values. They can have arbitrary dependence.}

\item{alpha}{Scalar (default \code{alpha = 0.05}) or vector of Type I error probabilities specifying the error rate to be spent or invested (Foster & Stine, 2008) on each of the \code{m = length(p)} null hypothesis tests, with total error rate \code{sum(alpha)} such as \code{sum(alpha) = 0.05} (which can be less than the desired total error rate if input \code{p} excludes \emph{p}-values from future online tests to be done later), as:\cr "Once we have spent this (total) error rate, it is gone" (Tukey, 1991, p.104).
\cr Input \code{alpha} helps define the random significance \emph{thresholds}, by: \cr \code{Delta_nu(r) = alpha * beta_nu(r)}, \cr for each ordered \emph{p}-value \code{p(r)} from \code{sort(p)}, based on a random probability measure \code{nu} on \verb{[0, m]} from a mixture of Dirichlet process, and on a positive reshaping parameter \code{beta_nu(.)}, used for MTP sensitivity analysis (see \strong{Details}).
\itemize{
\item \code{bnpMTP()} converts any scalar input \code{alpha} into the following vector input:\cr \code{alpha <- rep(alpha * (1 / m), m) = alpha * w}, which defines the thresholds of the \emph{Bonferroni} (1936) MTP for \code{m = length(p) = length(w)} tests, where each \emph{p}-value \code{p[i]} is assigned a \emph{standard weight} \code{w[i] = 1 / m}.
\item Input \code{alpha} can be specified as a vector: \code{alpha = alpha0 * w} for some small positive number \code{alpha0} (e.g., \code{alpha0 = 0.05}), which defines the significance thresholds \code{alpha} of the \emph{weighted Bonferroni} MTP, based on a prior distribution vector \code{w} representing the degree of belief for each of the \code{m = length(p)} null hypotheses (Genovese et al.2006), where \code{sum(w) = 1} (Tamhane & Gou 2022), or \code{sum(w) < 1} if \code{p} excludes \emph{p}-values from future online tests to be done later (Tian & Ramdas, 2021, Section 2).
\item Some alternatives for vector input \code{alpha} are defined by: the \emph{Šidák} (1967), \emph{Fallback} (Wiens & Dmitrienko, 2005), and \emph{Adaptive Discarding} MTPs for offline or online FWER control (Tian & Ramdas, 2021); and \emph{LORD} (Javanmard & Montanari 2018) and other \emph{generalized alpha investing} methods (Aharoni & Rosset, 2014) for online FDR control (Robertson et al. 2023).
}}

\item{N, mu}{Number of random samples drawn from the mixture of Dirichlet process (\code{DP(M, nu_0)}) prior distribution for the random probability measure \code{nu} defined on \verb{[0, m]}, with mass parameter \code{M} assigned an \code{Exponential(mu)} hyper-prior distribution with rate \code{mu}, where \code{m = length(p)}. Defaults: \code{N = 1000} and \code{mu = 1}.}
}
\value{
Output of the DP-MTP sensitivity analysis results, as a list containing the following objects:
\item{r.hat_nu}{A vector of \code{N} samples of the number, \code{r.hat_nu}, of the smallest \emph{p}-values (from input \code{p}) that are significant discoveries, based on \code{N} samples of the random probability measure \code{nu} defined on \verb{[0, m]} for \code{m = length(p)} hypothesis tests, with \code{nu} assigned the mixture of DP prior distribution.}
\item{Delta_nu.r,beta_nu.r}{Two \code{N}-by-\code{(m + 1)} matrices of \code{N} mixture of DP samples of the threshold function and the shape function for the sorted \emph{p}-values (\code{sort(p)}) in \code{colnames(Delta_nu.r)} and \code{colnames(beta_nu.r)}, respectively. Using \code{I = cbind(1:N,r.hat_nu+1)}, the \code{N} samples of threshold \code{Delta_nu(r.hat_nu)} and shape \code{beta_nu(r.hat_nu)} are obtained from \code{Delta_nu.r[I]} and \code{beta_nu.r[I]}.}
\item{Table}{A \code{3}-by-\code{(m + 1)} matrix reporting the probability of significance (\code{PrSig.p}) for each of the \code{m = length(p)} total \emph{p}-values in input \code{p} with respective error rate(s) \code{alpha}, based on the mixture DP prior. \cr For each ordered \emph{p}-value \code{p_(r)} from \code{sort(p)}, the \emph{probability of significance} is estimated by the proportion of \code{N} samples of \code{r.hat_nu} which satisfy inequality: \cr \code{p_(r) <= Delta_nu(r.hat_nu)}, for \code{r} = \verb{1,...,m = length(p)}. \cr The last column of the output \code{Table} shows the prior predictive \emph{p}-value from the global joint test that all \code{m = length(p)} null hypotheses are true; and their total spent error rate \code{sum(alpha)} and their \code{max(PrSig.p)}. This \emph{p}-value equals:
\cr \code{min(1 - PrSig.p) = 1 - max(PrSig.p)}
\cr                  \verb{=  mean(r.hat_nu == 0) = 1 - mean(r.hat_nu > 0)}
\cr based on the idea that the joint null hypothesis should be rejected if at least one of the \code{m} null hypotheses is rejected (Simes, 1986).}
}
\description{
Given inputs of \emph{p}-values \code{p} from \code{m = length(p)} hypothesis tests and their error rates \code{alpha}, this \code{R} package function \code{bnpMTP()} performs sensitivity analysis and uncertainty quantification for Multiple Testing Procedures (MTPs) based on a mixture of Dirichlet process (DP) prior distribution (Ferguson, 1973) supporting all MTPs providing Family-wise Error Rate (FWER) or False Discovery Rate (FDR) control for \emph{p}-values with arbitrary dependencies, e.g., due to tests performed on shared data and/or correlated variables, etc. From such an analysis, \code{bnpMTP()} outputs the distribution of the number of significant \emph{p}-values (discoveries); and a \emph{p}-value from a global joint test of all \code{m}  null hypotheses, based on the probability of significance (discovery) for each \emph{p}-value.\cr
The DP-MTP sensitivity analysis method can analyze a large number of \emph{p}-values obtained from any mix of null hypothesis testing procedures, including one-sample and/or multi-sample tests of: location, scale, higher moments, distribution, or symmetry; correlation, association, regression coefficients, odds ratios; change-points; runs; networks; classification; clustering; posterior distributions; model fit; outlyingness; and/or continuous hypothesis tests (e.g., performed on a realization of a random field); among other tests. Also, this sensitivity analysis method handles \emph{p}-values from traditional offline testing; and from \emph{online testing} performed on a stream of null hypotheses arriving one-by-one (or in blocks) over time (or asynchronously), where each test is based only on previous fixed test decisions and evidence against the current hypothesis, with unknown future data and total number of hypotheses being tested (potentially infinite) (Robertson, 2023).
\cr In any case, the DP-MTP sensitivity analysis method assumes that each \emph{p}-value follows a super-uniform distribution under the null hypothesis (i.e., either a Uniform(0,1) distribution under a calibrated test; or a stochastically larger distribution under a conservative test). More \strong{Details} about this method are below (run \code{bnpMTP} in \code{R} console to view code) and provided by Karabatsos (2025).
}
\details{
The Dirichlet process (\code{DP}) based MTP sensitivity analysis method (Karabatsos, 2025) assigns a mixture of \code{DP(M, nu_0)} prior distribution that flexibly supports the entire space of random probability measures \code{nu} defined on the interval \verb{[0, m]} for \code{m = length(p)} hypothesis tests, with \code{Exponential(mu)} hyper-prior distribution assigned to the \code{DP} mass parameter \code{M}, and with (mean) baseline probability measure (\code{nu_0}) defined by the Benjamini & Yekutieli (2001) MTP. In turn, this mixture DP prior also supports the space of all MTPs providing FWER or FDR control for \emph{p} values with arbitrary dependencies, because each of these MTPs can be uniquely characterized by a random probability measure \code{nu}, based on the \emph{shape function approach} to multiple hypothesis testing (Blanchard & Roquain, 2008, Sections 3.1-3.2; Lemma 3.2, Equation 6, pp.970–972, 976).

Specifically, the DP random probability measure, \code{nu}, drives the random number, \code{r.hat_nu}, of the smallest \emph{p}-values (from input \code{p} with \code{length(p) = m}) that are significant discoveries, defined via the following DP random \emph{step-up procedure} (using inequality <=):

\verb{r.hat_nu = max[r \\in \{0,1,...,m\} | p_(r)} <= \verb{alpha * beta_nu(r)]},

where for \code{r} = \verb{0,1,...,m}, the \code{p_(r)} (with \code{p_(0) := 0}) are the ordered \emph{p}-values (\code{sort(p)}) sorted in increasing order, with values of random significance \emph{thresholds}:\cr \code{Delta_nu(r) = alpha * beta_nu(r)},\cr based on a random \emph{shape function}:\cr \verb{beta_nu(r) = integral_0^r x d\{nu(x)\}} \cr which \emph{reshapes} (Ramdas et al. 2019, pp.2795-2796) or modifies \code{alpha} into new significance thresholds \code{Delta_nu(r)} to further account for arbitrary dependencies between \emph{p}-values.

Further details are provided by Karabatsos (2025), who illustrated this DP-MTP sensitivity analysis method on over twenty-eight thousand \emph{p}-values of different hypothesis tests performed on observations of 239 variables from a large dataset.
}
\examples{
#---------------------------------------------------------------------------------------
# Consider a classic data set in the field of multiple hypothesis testing procedures.
# Needleman (1979,Table 3) from yes/no responses to 11 Teachers' Behavioral survey items
# compared 58 children exposed to high lead and 100 children exposed to low lead levels;
# by p-values from 11 chi-square null hypothesis tests of equal group \% 'yes' responses;
# and a 2-tail p-value (0.02) from ANCOVA F-test of null hypothesis of equal group means
# in total sum score on the 11 items, while controlling for mother age at child's birth, 
# number of pregnancies & educational level; father's socioeconomic status; parental IQ.
#---------------------------------------------------------------------------------------

# Below, enter the vector of twelve p-values (and then run this R code line, below):
p        = c(0.003, 0.05, 0.05, 0.14, 0.08, 0.01, 0.04, 0.01, 0.05, 0.003, 0.003, 0.02)

# Below, name these p-values (then run the three R code lines, below):
names(p) = c( "Distractible"  , "Impersistent"  , "Dependent"   , "Disorganized" ,
              "Hyperactive"   , "Impulsive"     , "Frustrated"  , "Daydreamer"   ,
              "MissEzDirect"  , "MissSeqDirect" , "LowFunction" , "SumScore"     )

# Get results of DP-MTP sensitivity analysis of the p-values: (Run 2 code lines, below):
set.seed(123) # for reproducibility of results of Monte Carlo sampling done by bnpMTP()
Result   = bnpMTP( p = p , alpha = 0.05 )

# Show probability of significance for each of m = length(p) = 12 p-values in input 'p'
# based on mixture of DP(M, nu_0) prior; and prior predictive p-value from global test
# of all 12 null hypotheses, and their total error sum(alpha) (run R code line below):
Result$Table

# Summarize mixture of DP(M, nu_0) prior distribution of number of significant p-values:
quantile( Result$r.hat_nu )

#---------------------------------------------------------------------------------------
# Now suppose that the p-values were obtained from an online stream of hypothesis tests,
# with more hypothesis tests to be performed in the future (possibly infinite). 
# Accordingly, we specify the alpha vector based on p-value weights (w) defined 
# by the geometric distribution on {1,2,...} with 'success' probability 0.35,
# with sum(w) < 1 over the currently available twelve p-values in input p. 
#---------------------------------------------------------------------------------------

# Get results of DP-MTP sensitivity analysis of the p-values: (Run 5 code lines, below):
alpha0   = 0.05
w        = dgeom( ( 1 : length(p) ) - 1 , prob = 0.35 ) # specify p-value weights.
alpha    = alpha0 * w
set.seed(123) # for reproducibility of results of Monte Carlo sampling done by bnpMTP()
Online   = bnpMTP( p = p , alpha = alpha )

# Show probability of significance for each of m = length(p) = 12 p-values in input 'p'
# based on mixture of DP(M, nu_0) prior; and prior predictive p-value from global test
# of the 12 null hypotheses so far and their total error sum(alpha) (run line below):
Online$Table

# Summarize mixture of DP(M, nu_0) prior distribution of number of significant p-values:
quantile( Online$r.hat_nu )
}
\references{
Aharoni, E., and Rosset, S. (2014). Generalized alpha-investing: Definitions, optimality results and application to public databases. \emph{Journal of the Royal Statistical Society Series B}, \strong{76}, 771–794. \url{https://www.jstor.org/stable/24774568}

Benjamini, Y., and Yekutieli, D. (2001). The control of the false discovery rate in multiple testing under dependency. \emph{Annals of Statistics}, \strong{29}, 1165–1188. \url{https://www.jstor.org/stable/2674075}

Blanchard, G., and Roquain, E. (2008). Two simple sufficient conditions for FDR control. \emph{Electronic Journal of Statistics}, \strong{2}, 963–992. \code{DOI: 10.1214/08-EJS180}

Bonferroni, C. (1936). Teoria statistica delle classi e calcolo delle probabilità. \emph{Pubblicazioni del R Istituto Superiore di Scienze Economiche e Commerciali di Firenze}, \strong{8}, 3–62.

Ferguson, T. (1973). A Bayesian analysis of some nonparametric problems. \emph{Annals of Statistics}, \strong{1}, 209–230. \url{https://www.jstor.org/stable/2958008}

Foster, D., and Stine, R. (2008). Alpha-investing: A procedure for sequential control of expected false discoveries. \emph{Journal of the Royal Statistical Society, Series B}, \strong{70}, 429–444. \url{https://www.jstor.org/stable/20203833}

Genovese, C., Roeder, K., and Wasserman, L. (2006). False discovery control with \emph{p}-value weighting, \emph{Biometrika}, \strong{93}, 509–524. \url{https://www.jstor.org/stable/20441304}

Javanmard, A., and Montanari, A. (2018). Online rules for control of false discovery rate and false discovery exceedance. \emph{Annals of Statistics}, \strong{46}, 526–554. \url{https://www.jstor.org/stable/26542797}

Karabatsos, G. (2025). Bayesian nonparametric sensitivity analysis of multiple test procedures under dependence. \emph{Biometrical Journal}. \url{https://arxiv.org/abs/2410.08080}. Paper presented in the \emph{14th International Conference on Bayesian Nonparametrics} at UCLA on June 26, 2025.

Needleman, H., Gunnoe, C., Leviton, A., Reed, R., Presie, H., Maher, C., and Barret, P. (1979). Deficits in psychologic and classroom performance of children with elevated dentine lead levels. \emph{New England Journal of Medicine}, \strong{300}, 689–695. \url{https://www.nejm.org/doi/10.1056/NEJM197903293001301}

Ramdas, A., Barber, R., Wainwright, M., and Jordan, M. (2019). A unified treatment of multiple testing with prior knowledge using the \emph{p}-filter. \emph{Annals of Statistics}, \strong{47}, 2790–2821. \url{https://www.jstor.org/stable/26784046}

Robertson, D., Wason, J., and Ramdas, A. (2023). Online multiple hypothesis testing. \emph{Statistical Science}, \strong{38}, 557–575. \url{https://pmc.ncbi.nlm.nih.gov/articles/PMC7615519/}

Šidák, Z. (1967). Rectangular confidence regions for the means of multivariate normal distributions. \emph{Journal of the American Statistical Association}, \strong{62}, 626–633. \url{https://www.jstor.org/stable/2283989}

Simes, R. (1986). An improved Bonferroni procedure for multiple tests of significance. \emph{Biometrika}, \strong{73}, 751–754. \url{https://www.jstor.org/stable/2336545}

Tamhane, A., and Gou, J. (2022). Multiple test procedures based on \emph{p}-values. Chapter 2 of \emph{Handbook of Multiple Comparisons}, by X. Cui, T. Dickhaus, Y. Ding, and J. Hsu (Eds.). CRC Press.

Tian, J., and Ramdas, A. (2021). Online control of the familywise error rate. \emph{Statistical Methods in Medical Research}, \strong{30}, 976–993. \url{https://pubmed.ncbi.nlm.nih.gov/33413033/}

Tukey, J. (1991). The philosophy of multiple comparisons. \emph{Statistical Science}, \strong{6}, 100–116. \url{https://www.jstor.org/stable/2245714}

Wiens, B., and Dmitrienko, A. (2005). The fallback procedure for evaluating a single family of hypotheses. \emph{Journal of Biopharmaceutical Statistics}, \strong{15}, 929–942. \url{https://www.tandfonline.com/doi/full/10.1080/10543400500265660}
}
