% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cs_distribution.R
\name{cs_distribution}
\alias{cs_distribution}
\title{Distribution-Based Analysis of Clinical Significance}
\usage{
cs_distribution(
  data,
  id,
  time,
  outcome,
  group = NULL,
  pre = NULL,
  post = NULL,
  reliability = NULL,
  reliability_post = NULL,
  better_is = c("lower", "higher"),
  rci_method = c("JT", "GLN", "HLL", "EN", "NK", "HA", "HLM"),
  significance_level = 0.05
)
}
\arguments{
\item{data}{A tidy data frame}

\item{id}{Participant ID}

\item{time}{Time variable}

\item{outcome}{Outcome variable}

\item{group}{Grouping variable (optional)}

\item{pre}{Pre measurement (only needed if the time variable contains more
than two measurements)}

\item{post}{Post measurement (only needed if the time variable contains more
than two measurements)}

\item{reliability}{The instrument's reliability estimate. If you selected the
NK method, the here specified reliability will be the instrument's pre
measurement reliability. Not needed for the HLM method.}

\item{reliability_post}{The instrument's reliability at post measurement
(only needed for the NK method)}

\item{better_is}{Which direction means a better outcome for the used
instrument? Available are
\itemize{
\item \code{"lower"} (lower outcome scores are desirable, the default) and
\item \code{"higher"} (higher outcome scores are desirable)
}}

\item{rci_method}{Clinical significance method. Available are
\itemize{
\item \code{"JT"} (Jacobson & Truax, 1991, the default)
\item \code{"GLN"} (Gulliksen, Lord, and Novick; Hsu, 1989, Hsu, 1995)
\item \code{"HLL"} (Hsu, Linn & Nord; Hsu, 1989)
\item \code{"EN"} (Edwards & Nunnally; Speer, 1992)
\item \code{"NK"} (Nunnally & Kotsch, 1983), requires a reliability estimate at post
measurement. If this is not supplied, reliability and reliability_post are
assumed to be equal
\item \code{"HA"} (Hageman & Arrindell, 1999)
\item \code{"HLM"} (Hierarchical Linear Modeling; Raudenbush & Bryk, 2002),
requires at least three measurements per patient
}}

\item{significance_level}{Significance level alpha, defaults to \code{0.05}. If
you choose the \code{"HA"} method, this value corresponds to the maximum risk of
misclassification}
}
\value{
An S3 object of class \code{cs_analysis} and \code{cs_distribution}
}
\description{
\code{cs_distribution()} can be used to determine the clinical
significance of intervention studies employing the distribution-based
approach. For this, the reliable change index is estimated from the
provided data and a known reliability estimate which indicates, if an
observed individual change is likely to be greater than the measurement
error inherent for the used instrument. In this case, a reliable change is
defined as clinically significant. Several methods for calculating this RCI
can be chosen.
}
\section{Computational details}{
 From the provided data, a region of change is
calculated in which an individual change may likely be due to an inherent
measurement of the used instrument. This concept is also known as the
minimally detectable change (MDC).
}

\section{Categories}{
 Each individual's change may then be categorized into
one of the following three categories:
\itemize{
\item Improved, the change is greater than the RCI in the beneficial direction
\item Unchanged, the change is within a region that may attributable to
measurement error
\item Deteriorated, the change is greater than the RCI, but in the
disadvantageous direction
}

Most of these methods are developed to deal with data containing two
measurements per individual, i.e., a pre intervention and post intervention
measurement. The Hierarchical Linear Modeling (\code{rci_method = "HLM"}) method
can incorporate data for multiple measurements an can thus be used only
for at least three measurements per participant.
}

\section{Data preparation}{
 The data set must be tidy, which corresponds to a
long data frame in general. It must contain a patient identifier which must
be unique per patient. Also, a column containing the different measurements
and the outcome must be supplied. Each participant-measurement combination
must be unique, so for instance, the data must not contain two "After"
measurements for the same patient.

Additionally, if the measurement column contains only two values, the first
value based on alphabetical, numerical or factor ordering will be used as
the \code{pre} measurement. For instance, if the column contains the
measurements identifiers \code{"pre"} and \code{"post"} as strings, then \code{"post"}
will be sorted before \code{"pre"} and thus be used as the \code{"pre"} measurement.
The function will throw a warning but generally you may want to explicitly
define the \code{"pre"} and \code{"post"} measurement with arguments \code{pre} and
\code{post}. In case of more than two measurement identifiers, you have to
define \code{pre} and \code{post} manually since the function does not know what your
pre and post intervention measurements are.

If your data is grouped, you can specify the group by referencing the
grouping variable (see examples below). The analysis is then run for every
group to compare group differences.
}

\examples{
antidepressants |>
  cs_distribution(patient, measurement, mom_di, reliability = 0.80)


# Turn off the warning by providing the pre measurement time
cs_results <- antidepressants |>
  cs_distribution(
    patient,
    measurement,
    mom_di,
    pre = "Before",
    reliability = 0.80
  )

summary(cs_results)
plot(cs_results)


# If you use data with more than two measurements, you always have to define a
# pre and post measurement
cs_results <- claus_2020 |>
  cs_distribution(
    id,
    time,
    hamd,
    pre = 1,
    post = 4,
    reliability = 0.80
  )

cs_results
summary(cs_results)
plot(cs_results)


# Set the rci_method argument to change the RCI method
cs_results_ha <- claus_2020 |>
  cs_distribution(
    id,
    time,
    hamd,
    pre = 1,
    post = 4,
    reliability = 0.80,
    rci_method = "HA"
  )

cs_results_ha
summary(cs_results_ha)
plot(cs_results_ha)


# Group the analysis by providing a grouping variable
cs_results_grouped <- claus_2020 |>
  cs_distribution(
    id,
    time,
    hamd,
    pre = 1,
    post = 4,
    group = treatment,
    reliability = 0.80
  )

cs_results_grouped
summary(cs_results_grouped)
plot(cs_results_grouped)


# Use more than two measurements
cs_results_hlm <- claus_2020 |>
  cs_distribution(
    id,
    time,
    hamd,
    rci_method = "HLM"
  )

cs_results_hlm
summary(cs_results_hlm)
plot(cs_results_hlm)
}
\references{
\itemize{
\item Jacobson, N. S., & Truax, P. (1991). Clinical significance: A statistical approach to defining meaningful change in psychotherapy research. Journal of Consulting and Clinical Psychology, 59(1), 12–19. https://doi.org/10.1037//0022-006X.59.1.12
\item Hsu, L. M. (1989). Reliable changes in psychotherapy: Taking into account regression toward the mean. Behavioral Assessment, 11(4), 459–467.
\item Hsu, L. M. (1995). Regression toward the mean associated with measurement error and the identification of improvement and deterioration in psychotherapy. Journal of Consulting and Clinical Psychology, 63(1), 141–144. https://doi.org/10.1037//0022-006x.63.1.141
\item Speer, D. C. (1992). Clinically significant change: Jacobson and Truax (1991) revisited. Journal of Consulting and Clinical Psychology, 60(3), 402–408. https://doi.org/10.1037/0022-006X.60.3.402
\item Nunnally, J. C., & Kotsch, W. E. (1983). Studies of individual subjects: Logic and methods of analysis. British Journal of Clinical Psychology, 22(2), 83–93. https://doi.org/10.1111/j.2044-8260.1983.tb00582.x
\item Hageman, W. J., & Arrindell, W. A. (1999). Establishing clinically significant change: increment of precision and the distinction between individual and group level analysis. Behaviour Research and Therapy, 37(12), 1169–1193. https://doi.org/10.1016/S0005-7967(99)00032-7
\item Raudenbush, S. W., & Bryk, A. S. (2002). Hierarchical Linear Models - Applications and Data Analysis Methods (2nd ed.). Sage Publications.
}
}
\seealso{
Main clinical signficance functions
\code{\link{cs_anchor}()},
\code{\link{cs_combined}()},
\code{\link{cs_percentage}()},
\code{\link{cs_statistical}()}
}
\concept{main}
