% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/update_rates_and_regimes_for_focal_time.R
\name{update_rates_and_regimes_for_focal_time}
\alias{update_rates_and_regimes_for_focal_time}
\title{Update diversification rates/regimes mapped on a phylogeny up to a given time in the past}
\usage{
update_rates_and_regimes_for_focal_time(
  BAMM_object,
  focal_time,
  update_rates = TRUE,
  update_regimes = TRUE,
  update_tree = FALSE,
  update_plot = FALSE,
  update_all_elements = FALSE,
  keep_tip_labels = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{BAMM_object}{Object of class \code{"bammdata"}, typically generated with \code{\link[=prepare_diversification_data]{prepare_diversification_data()}},
that contains a phylogenetic tree and associated diversification rate mapping across selected posterior samples.
The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{focal_time}{Numerical. The time, in terms of time distance from the present,
at which the tree and rate mapping must be cut. It must be smaller than the root age of the phylogeny.}

\item{update_rates}{Logical. Specify whether diversification rates stored in
\verb{$tipLambda} (speciation) and \verb{$tipMu} (extinction) must be updated to summarize
rates found along branches at \code{focal_time}. Default is \code{TRUE}.}

\item{update_regimes}{Logical. Specify whether diversification regimes stored in
\verb{$tipStates} must be updated to summarize regimes found along branches at \code{focal_time}.
Default is \code{TRUE}.}

\item{update_tree}{Logical. Specify whether to update the phylogeny such as
all branches that are younger than the \code{focal_time} are cut-off. Default is \code{FALSE}.}

\item{update_plot}{Logical. Specify whether to update the phylogeny AND the elements
used by \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}} to plot diversification rates on the phylogeny
such as all branches that are younger than the \code{focal_time} are cut-off. Default is \code{FALSE}.
If set to \code{TRUE}, it will override the \code{update_tree} parameter and update the phylogeny.}

\item{update_all_elements}{Logical. Specify whether to update all the elements in the object, including
rates/regimes/phylogeny/elements for \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}}/all other elements. Default is \code{FALSE}.
If set to \code{TRUE}, it will override other \verb{update_*} parameters and update all elements.}

\item{keep_tip_labels}{Logical. Should terminal branches with a single descendant tip
retain their initial \code{tip.label} on the updated phylogeny and diversification rate mapping?
Default is \code{TRUE}. If set to \code{FALSE}, the tipward node ID will be used as label for all tips.}

\item{verbose}{Logical. Should progression be displayed? A message will be printed for every batch of
100 BAMM posterior samples updated. Default is \code{TRUE}.}
}
\value{
The function returns a list as an updated \code{BAMM_object} of class \code{"bammdata"}.

Phylogeny-related elements used to plot a phylogeny with \code{\link[ape:plot.phylo]{ape::plot.phylo()}}:
\itemize{
\item \verb{$edge} Matrix of integers. Defines the tree topology by providing rootward and tipward node ID of each edge.
\item \verb{$Nnode} Integer. Number of internal nodes.
\item \verb{$tip.label} Vector of character strings. Labels of all tips, including fossils older than \code{focal_time} if present.
\itemize{
\item If \code{keep_tip_labels = TRUE}, cut-off branches with a single descendant tip retain their initial \code{tip.label}.
\item If \code{keep_tip_labels = FALSE}, all cut-off branches are labeled using their tipward node ID.
}
\item \verb{$edge.length} Vector of numerical. Length of edges/branches.
\item \verb{$node.label} Vector of character strings. Labels of all internal nodes. (Present only if present in the initial \code{BAMM_object})
}

BAMM internal elements used for tree exploration:
\itemize{
\item \verb{$begin} Vector of numerical. Absolute time since root of edge/branch start (rootward).
\item \verb{$end} Vector of numerical.  Absolute time since root of edge/branch end (tipward).
\item \verb{$downseq} Vector of integers. Order of node visits when using a pre-order tree traversal.
\item \verb{$lastvisit} ID of the last node visited when starting from the node in the corresponding position in \verb{$downseq}.
}

BAMM elements summarizing diversification data:
\itemize{
\item \verb{$numberEvents} Vector of integer. Number of events/macroevolutionary regimes (k+1) recorded in each posterior configuration. k = number of shifts.
\item \verb{$eventData} List of data.frames. One per posterior sample. Records shift events and macroevolutionary regimes parameters. 1st line = Background root regime.
\item \verb{$eventVectors} List of integer vectors. One per posterior sample. Record regime ID per branches.
\item \verb{$tipStates} List of named integer vectors. One per posterior sample. Record regime ID per tips present at \code{focal_time}. Updated if \code{update_regimes = TRUE}.
\item \verb{$tipLambda} List of named numerical vectors. One per posterior sample. Record speciation rates per tips present at \code{focal_time}. Updated if \code{update_rates = TRUE}.
\item \verb{$tipMu} List of named numerical vectors. One per posterior sample. Record extinction rates per tips present at \code{focal_time}. Updated if \code{update_rates = TRUE}.
\item \verb{$eventBranchSegs} List of matrix of numerical. One per posterior sample. Record regime ID per segments of branches.
\item \verb{$meanTipLambda} Vector of named numerical. Mean tip speciation rates across all posterior configurations of tips present at \code{focal_time} (does not includes older fossils).
\item \verb{$meanTipMu} Vector of named numerical. Mean tip extinction rates across all posterior configurations of tips present at \code{focal_time} (does not includes older fossils).
\item \verb{$type} Character string. Set the type of data modeled with BAMM. Should be "diversification".
}

Additional elements providing key information for downstream analyses:
\itemize{
\item \verb{$expectedNumberOfShifts} Integer. The expected number of regime shifts used to set the prior in BAMM.
\item \verb{$MSP_tree} Object of class \code{phylo}. List of 4 elements duplicating information from the Phylogeny-related elements above,
except \verb{$MSP_tree$edge.length} is recording the Marginal Shift Probability of each branch (i.e., the probability of a regime shift to occur along each branch)
whose origin is older that \code{focal_time}.
\item \verb{$MAP_indices} Vector of integers. The indices of the Maximum A Posteriori probability (MAP) configurations among the posterior samples.
\item \verb{$MAP_BAMM_object}. List of 18 elements of class \verb{"bammdata" recording the mean rates and regime shift locations found across the Maximum A Posteriori probability (MAP) configuration. All BAMM elements summarizing diversification data holds a single entry describing this the mean diversification history, updated for the }focal_time`.
\item \verb{$MSC_indices} Vector of integers. The indices of the Maximum Shift Credibility (MSC) configurations among the posterior samples.
\item \verb{$MSC_BAMM_object} List of 18 elements of class \verb{"bammdata" recording the mean rates and regime shift locations found across the Maximum Shift Credibility (MSC) configurations. All BAMM elements summarizing diversification data holds a single entry describing this mean diversification history, updated for the }focal_time`.
}

New elements added to provide update information:
\itemize{
\item \verb{$root_age} Integer. Stores the age of the root of the tree.
\item \verb{$nodes_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_node_ID} to the \code{initial_node_ID}. Each row is a node.
\item \verb{$initial_nodes_ID} Vector of character strings. Provides the initial ID of internal nodes. Used to plot internal node IDs as labels with \code{\link[ape:nodelabels]{ape::nodelabels()}}.
\item \verb{$edges_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_edge_ID} to the \code{initial_edge_ID}. Each row is an edge/branch.
\item \verb{$initial_edges_ID} Vector of character strings. Provides the initial ID of edges/branches. Used to plot edge/branch IDs as labels with \code{\link[ape:nodelabels]{ape::edgelabels()}}.
\item \verb{$dtrates} List of three elements.
\itemize{
\item 1/ \verb{$dtrates$tau} Numerical. Resolution factor describing the fraction of each segment length used in \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}}
compared to the full depth of the initial tree (i.e., the root_age)
\item 2/ \verb{$dtrates$rates} List of two numerical vectors. Speciation and extinction rates along segments used by \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}}.
\item 3/ \verb{$dtrates$tmat} Matrix of numerical. Start and end times of segments in term of distance to the root.
}
\item \verb{$initial_colorbreaks} List of three vectors of numerical. Rate values of the percentiles delimiting the bins for mapping rates to colors with \code{\link[BAMMtools:plot]{BAMMtools::plot.bammdata()}}.
Each element provides values for different type of rates (\verb{$speciation}, \verb{$extinction}, \verb{$net_diversification}).
\item \verb{$focal_time} Integer. The time, in terms of time distance from the present, at which the rates/regimes were extracted and the tree was eventually cut.
}
}
\description{
Updates an object of class \code{"bammdata"} to obtain the diversification rates/regimes
found along branches at a specific time in the past (i.e. the \code{focal_time}).
Optionally, the function can update the object to display a mapped phylogeny such as
branches overlapping the \code{focal_time} are shorten to the \code{focal_time}.
}
\details{
The object of class \code{"bammdata"} (\code{BAMM_object}) is cut-off at a specific time in the past
(i.e. the \code{focal_time}) and the current diversification rate values of the overlapping edges/branches are extracted.

----- Update diversification rate data -----

If \code{update_rates = TRUE}, diversification rates of branches overlapping with \code{focal_time}
will be updated. Each cut-off branches form a new tip present at \code{focal_time} with updated associated
diversification rate data. Fossils older than \code{focal_time} do not yield any data.
\itemize{
\item \verb{$tipLambda} contains speciation rates.
\item \verb{$tipMu} contains extinction rates.
}

If \code{update_regimes = TRUE}, diversification regimes of branches overlapping with \code{focal_time}
will be updated. Each cut-off branches form a new tip present at \code{focal_time} with updated associated
diversification regime ID found in \verb{$tipStates}. Fossils older than \code{focal_time} do not yield any data.

----- Update the phylogeny -----

If \code{update_tree = TRUE}, elements defining the phylogeny, as in an \code{"phylo"} object
will be updated such as all branches that are younger than the \code{focal_time} are cut-off:
\itemize{
\item \verb{$edge} defines the tree topology.
\item \verb{$Nnode} defines the number of internal nodes.
\item \verb{$tip.label} provides the labels of all tips, including fossils older than \code{focal_time} if present.
\item \verb{$edge.length} provides length of all branches.
\item \verb{$node.label} provides the labels of all internal nodes. (Optional)
}

----- Update the plot from \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}} -----

If \code{update_plot = TRUE}, elements used to plot diversification rates on the phylogeny
using \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}} will be updated such as all branches that are younger
than the \code{focal_time} are cut-off:
\itemize{
\item \verb{$begin} provides absolute time since root of edge/branch start (rootward).
\item \verb{$end} provides absolute time since root of edge/branch end (tipward).
\item \verb{$eventVectors} provides regime membership per branches in each posterior sample configuration.
\item \verb{$eventBranchSegs} provides regime membership per segments of branches in each posterior sample configuration.
\item \verb{$dtrates} provides mean speciation and extinction rates along segments of branches, and resolution fraction (tau) describing
the fraction of each segment length compared to the full depth of the initial tree (i.e., the root_age).
}
}
\examples{
# ----- Example 1: Extant whales (87 taxa) ----- #

## Load the BAMM_object summarizing 1000 posterior samples of BAMM
data(whale_BAMM_object, package = "deepSTRAPP")

## Set focal-time to 5 My
focal_time = 5

\donttest{ # (May take several minutes to run)
## Update the BAMM object
whale_BAMM_object_5My <- update_rates_and_regimes_for_focal_time(
   BAMM_object = whale_BAMM_object,
   focal_time = 5,
   update_rates = TRUE, update_regimes = TRUE,
   update_tree = TRUE, update_plot = TRUE,
   update_all_elements = TRUE,
   keep_tip_labels = TRUE,
   verbose = TRUE)

# Add "phylo" class to be compatible with phytools::nodeHeights()
class(whale_BAMM_object) <- unique(c(class(whale_BAMM_object), "phylo"))
root_age <- max(phytools::nodeHeights(whale_BAMM_object)[,2])
# Remove temporary "phylo" class
class(whale_BAMM_object) <- setdiff(class(whale_BAMM_object), "phylo")

## Plot initial BAMM_object for t = 0 My
plot_BAMM_rates(whale_BAMM_object, add_regime_shifts = TRUE,
                labels = TRUE, legend = TRUE, cex = 0.5,
                par.reset = FALSE) # Keep plotting parameters in memory to use abline().
abline(v = root_age - focal_time,
      col = "red", lty = 2, lwd = 2)

## Plot updated BAMM_object for t = 5 My
plot_BAMM_rates(whale_BAMM_object_5My, add_regime_shifts = TRUE,
                labels = TRUE, legend = TRUE, cex = 0.8) }

# ----- Example 2: Extant Ponerinae (1,534 taxa) ----- #

if (deepSTRAPP::is_dev_version())
{
 ## Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Set focal-time to 10 My
 focal_time = 10

 \donttest{ # (May take several minutes to run)
 ## Update the BAMM object
 Ponerinae_BAMM_object_10My <- update_rates_and_regimes_for_focal_time(
    BAMM_object = Ponerinae_BAMM_object,
    focal_time = focal_time,
    update_rates = TRUE, update_regimes = TRUE,
    update_tree = TRUE, update_plot = TRUE,
    update_all_elements = TRUE,
    keep_tip_labels = TRUE,
    verbose = TRUE) }

 ## Load results to save time
 data(Ponerinae_BAMM_object_10My, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Extract root age
 # Add temporarily the "phylo" class to be compatible with phytools::nodeHeights()
 class(Ponerinae_BAMM_object) <- unique(c(class(Ponerinae_BAMM_object), "phylo"))
 root_age <- max(phytools::nodeHeights(Ponerinae_BAMM_object)[,2])
 # Remove temporary "phylo" class
 class(Ponerinae_BAMM_object) <- setdiff(class(Ponerinae_BAMM_object), "phylo")

 ## Plot diversification rates on the initial tree
 plot_BAMM_rates(Ponerinae_BAMM_object,
                 legend = TRUE, labels = FALSE)
 abline(v = root_age - focal_time,
        col = "red", lty = 2, lwd = 2)

 ## Plot diversification rates and regime shifts on the updated tree (cut-off for 10 My)
 # Keep the initial color scheme
 plot_BAMM_rates(Ponerinae_BAMM_object_10My, legend = TRUE, labels = FALSE,
                 colorbreaks = Ponerinae_BAMM_object_10My$initial_colorbreaks$net_diversification)

 # Use a new color scheme mapped on the new distribution of rates
 plot_BAMM_rates(Ponerinae_BAMM_object_10My, legend = TRUE, labels = FALSE)
}

}
\seealso{
\code{\link[=cut_phylo_for_focal_time]{cut_phylo_for_focal_time()}} \code{\link[=plot_BAMM_rates]{plot_BAMM_rates()}}
}
\author{
Maël Doré
}
