% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ds.glm.R
\name{ds.glm}
\alias{ds.glm}
\title{Fits Generalized Linear Model}
\usage{
ds.glm(
  formula = NULL,
  data = NULL,
  family = NULL,
  offset = NULL,
  weights = NULL,
  checks = FALSE,
  maxit = 20,
  CI = 0.95,
  viewIter = FALSE,
  viewVarCov = FALSE,
  viewCor = FALSE,
  datasources = NULL
)
}
\arguments{
\item{formula}{an object of class formula describing
the model to be fitted. For more information see 
\strong{Details}.}

\item{data}{a character string specifying the name of an (optional) data frame that contains
all of the variables in the GLM formula.}

\item{family}{identifies the error distribution function to use in
the model. 
This can be set as \code{"gaussian"}, \code{"binomial"} and \code{"poisson"}. 
For more information see \strong{Details}.}

\item{offset}{a character string specifying the name of a variable to be used as
an offset. \code{ds.glm} does not allow an offset vector to be
written directly into the GLM formula. For more information see \strong{Details}.}

\item{weights}{a character string specifying the name of a variable containing
prior regression weights for the fitting process. 
\code{ds.glm} does not allow a weights vector to be
written directly into the GLM formula.}

\item{checks}{logical. If TRUE \code{ds.glm} checks the structural integrity 
of the model. Default FALSE. For more information see \strong{Details}.}

\item{maxit}{a numeric scalar denoting the maximum number of iterations that are permitted
before \code{ds.glm} declares that the model has failed to converge.}

\item{CI}{a numeric value specifying the confidence interval. Default \code{0.95}.}

\item{viewIter}{logical. If TRUE the results of the intermediate iterations are 
printed. If FALSE only final results are shown. Default FALSE.}

\item{viewVarCov}{logical. If TRUE the variance-covariance matrix
of parameter estimates is returned. Default FALSE.}

\item{viewCor}{logical. If TRUE the correlation matrix of
parameter estimates is returned. Default FALSE.}

\item{datasources}{a list of \code{\link[DSI]{DSConnection-class}} objects obtained after login. 
If the \code{datasources} argument is not specified
the default set of connections will be used: see \code{\link[DSI]{datashield.connections_default}}.}
}
\value{
Many of the elements of the output list returned by \code{ds.glm} are 
equivalent to those returned by the \code{glm()} function in native R. However,
potentially disclosive elements
such as individual-level residuals and linear predictor values are blocked. 
In this case, only non-disclosive elements are returned from each study separately.

The list of elements returned by \code{ds.glm} is mentioned below:

\code{Nvalid}: total number of valid observational units across all studies.

\code{Nmissing}: total number of observational units across all studies with at least
                            one data item missing.

\code{Ntotal}: total of observational units across all studies, the  
                          sum of valid and missing units.

\code{disclosure.risk}: risk of disclosure,
                                   the value 1 indicates that one of the disclosure traps 
                                   has been triggered in that study.

\code{errorMessage}: explanation for any errors or disclosure risks identified.

\code{nsubs}: total number of observational units used by \code{ds.glm} function. 
                       \code{nb} usually is the same as \code{nvalid}.

\code{iter}: total number of iterations before convergence achieved.

\code{family}: error family and link function.

\code{formula}: model formula, see description of formula as an input parameter (above).

\code{coefficients}: a matrix with 5 columns:
   \describe{
   \item{First}{: the names of all of the regression parameters (coefficients) in the model}
   \item{second}{: the estimated values}
   \item{third}{: corresponding standard errors of the estimated values}
   \item{fourth}{: the ratio of estimate/standard error}.
   \item{fifth}{: the p-value treating that as a standardised normal deviate}
    }

\code{dev}: residual deviance.

\code{df}: residual degrees of freedom. \code{nb} residual degrees of freedom + number of
   parameters in model = \code{nsubs}.

\code{output.information}: reminder to the user that there 
                                     is more information at the top of the output.

Also, the estimated coefficients and standard errors expanded with estimated confidence intervals
   with \% coverage specified by \code{ci} argument are returned. 
   For the poisson model,
   the output is generated on the scale of the linear predictor (log rates and log rate ratios) 
   and the natural scale after exponentiation (rates and rate ratios).
}
\description{
Fits a Generalized Linear Model (GLM) on data from single or multiple sources
on the server-side.
}
\details{
Fits a GLM on data from a single source or multiple sources on the server-side.
In the latter case, the data are co-analysed (when using \code{ds.glm}) 
by using an approach that is mathematically equivalent to placing all individual-level
data from all sources in one central warehouse and analysing those data using the conventional
\code{glm()} function in R. In this situation marked heterogeneity between sources should be corrected
(where possible) with fixed effects. For example, if each study in a (binary) logistic regression
analysis has an independent intercept, it is equivalent to allowing each study to have a
different baseline risk of disease. This may also be viewed as being an IP (individual person)
meta-analysis with fixed effects.


In \code{formula} most shortcut notation for formulas allowed under R's standard \code{glm()}
function is also allowed by \code{ds.glm}. 

Many GLMs can be fitted very simply using a formula such as:

\eqn{y~a+b+c+d}

which simply means fit a GLM with \code{y} as the outcome variable and 
\code{a}, \code{b}, \code{c} and \code{d} as covariates. 
By default all such models also include an intercept (regression constant) term.

Instead, if you need to fit a more complex
model, for example:

 \eqn{EVENT~1+TID+SEXF*AGE.60}

In the above model the outcome variable is \code{EVENT} 
and the  covariates 
\code{TID} (factor variable with level values between 1 and 6 denoting the period time), 
\code{SEXF} (factor variable denoting sex)
and \code{AGE.60} (quantitative variable representing age-60 in years). 
The term \code{1} forces
the model to include an intercept term, in contrast if you use the term \code{0} the 
intercept term is removed. The \code{*} symbol  between \code{SEXF} and \code{AGE.60}
means fit all possible main effects and interactions for and between those two covariates.
 This takes the value 0 in all males \code{0 * AGE.60} 
 and in females  \code{1 * AGE.60}.
 This model is in example 1 of  the section \strong{Examples}. In this case the logarithm of 
 the survival time is added as an offset (\code{log(survtime)}).  
 

In the \code{family} argument can be specified three types of models to fit:

 \describe{
   \item{\code{"gaussian"}}{: conventional linear model with normally distributed errors} 
   \item{\code{"binomial"}}{: conventional unconditional logistic regression model}
   \item{\code{"poisson"}}{: Poisson regression model which is the most used in survival analysis. 
    The model used Piecewise Exponential Regression (PER) which typically closely approximates
    Cox regression in its main estimates and standard errors.}
}


At present the gaussian family is automatically coupled with
an \code{identity} link function, the binomial family with a
\code{logistic} link function and the poisson family with a \code{log} link function. 


The \code{data} argument avoids you having to specify the name of the
data frame in front of each covariate in the formula. 
For example, if the data frame is called \code{DataFrame} you
avoid having to write: \eqn{DataFrame\$y ~ DataFrame\$a + DataFrame\$b + DataFrame\$c + DataFrame\$d}

The \code{checks} argument verifies that the variables in the model are all defined (exist) 
on the server-side at every study
and that they have the correct characteristics required to fit the model. 
It is suggested to make \code{checks} argument TRUE if an unexplained
 problem in the model fit is encountered because the running process takes several minutes.

In \code{maxit}  Logistic regression and Poisson regression
models can require many iterations, particularly if the starting value of the
regression constant is far away from its actual value that the GLM
is trying to estimate. In consequence we often set \code{maxit=30}
but depending on the nature of the models you wish to fit, you may wish
to be alerted much more quickly than this if there is a delay in convergence, 
or you may wish to all more iterations.


Privacy protected iterative fitting of a GLM is explained here:

(1) Begin with a guess for the coefficient vector to start iteration 1 (let's call it
\code{beta.vector[1]}). Using \code{beta.vector[1]}, run iteration 1 with each source
calculating the resultant score vector (and information matrix) generated
by its data - given \code{beta.vector[1]} -
as the sum of the score vector components (and the sum of the components of the
information matrix) derived from each individual data record in that source. NB in most models
the starting values in \code{beta.vector[1]} are set to be zero for all parameters.

(2) Transmit the resultant score vector and information matrix from each source
back to the clientside
server (CS) at the analysis centre. Let's denote
\code{SCORE[1][j]} and \code{INFORMATION.MATRIX[1][j]} as the
score vector and information matrix generated by study \code{j} at the end of the 1st iteration.

(3) CS sums the score vectors, and equivalently the information matrices, across all studies
(i.e. \code{j = 1:S}, where \code{S} is the number of studies). Note that,
given \code{beta.vector[1]}, this gives precisely the same final sums
for the score
vectors and information matrices as would have been obtained if all data had been in one
central warehoused database and the overall score vector and information matrix at the end of
the first iteration had been calculated
(as is standard) by simply summing across all individuals. The only difference is that
instead of directly adding all values across
all individuals, we first sum across all individuals in each data source and
then sum those study
totals across all studies - i.e. this generates the same ultimate sums

(4) CS then calculates \code{sum(SCORES)\%*\% inverse(sum(INFORMATION.MATRICES))} -
heuristically this may be
viewed as being "the sum of the score vectors divided (NB 'matrix division') by the sum of the
information matrices". If one uses the conventional algorithm (IRLS)
to update generalized linear models from iteration to iteration this quantity happens to be
precisely the vector to be added to the
current value of beta.vector (i.e. \code{beta.vector[1]}) to obtain
\code{beta.vector[2]} which is the improved estimate of the beta.vector to be used in iteration 2.
This updating algorithm is often  called the IRLS (Iterative Reweighted Least
Squares) algorithm
- which is closely related to the Newton
Raphson approach but uses the expected information rather than
the observed information.

(5) Repeat steps (2)-(4) until the model converges (using the standard R
convergence criterion).
NB An alternative way to coherently pool the glm across multiple sources is to fit each
glm to completion (i.e. multiple iterations until convergence) in each source and then return
the final parameter estimates and standard errors to the CS where they could be pooled using
study-level meta-analysis. An alternative function ds.glmSLMA allows you to do this.
It will fit the glms to completion
in each source and return the final estimates and standard errors (rather than score vectors
and information matrices). It will then rely on functions in the
R package metafor to meta-analyse the key parameters.


Server functions called: \code{glmDS1} and \code{glmDS2}
}
\examples{
\dontrun{

 ## Version 6, for version 5 see Wiki
  # Connecting to the Opal servers
  
  require('DSI')
  require('DSOpal')
  require('dsBaseClient')
  
  # Example 1: Fitting GLM for survival analysis
  # For this analysis we need to load survival data from the server 
  
  builder <- DSI::newDSLoginBuilder()
  builder$append(server = "study1", 
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "SURVIVAL.EXPAND_NO_MISSING1", driver = "OpalDriver")
  builder$append(server = "study2", 
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "SURVIVAL.EXPAND_NO_MISSING2", driver = "OpalDriver")
  builder$append(server = "study3",
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "SURVIVAL.EXPAND_NO_MISSING3", driver = "OpalDriver")
  logindata <- builder$build()
  
  # Log onto the remote Opal training servers
  connections <- DSI::datashield.login(logins = logindata, assign = TRUE, symbol = "D") 
  
  # Fit the GLM 
  
  # make sure that the outcome is numeric 
  ds.asNumeric(x.name = "D$cens",
               newobj = "EVENT",
               datasources = connections)
               
  # convert time id variable to a factor 
               
  ds.asFactor(input.var.name = "D$time.id",
              newobj = "TID",
              datasources = connections)
              
  # create in the server-side the log(survtime) variable
         
  ds.log(x = "D$survtime",
         newobj = "log.surv",
         datasources = connections)
  
  ds.glm(formula = EVENT ~ 1 + TID + female * age.60,
         data = "D",
         family = "poisson", 
         offset = "log.surv",
         weights = NULL,
         checks = FALSE,
         maxit = 20,
         CI = 0.95,
         viewIter = FALSE,
         viewVarCov = FALSE,
         viewCor = FALSE,
         datasources = connections)
         
  # Clear the Datashield R sessions and logout
  datashield.logout(connections) 
  
  # Example 2: run a logistic regression without interaction
  # For this example we are going to load another dataset  
  
  builder <- DSI::newDSLoginBuilder()
  builder$append(server = "study1", 
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "CNSIM.CNSIM1", driver = "OpalDriver")
  builder$append(server = "study2", 
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "CNSIM.CNSIM2", driver = "OpalDriver")
  builder$append(server = "study3",
                 url = "http://192.168.56.100:8080/", 
                 user = "administrator", password = "datashield_test&", 
                 table = "CNSIM.CNSIM3", driver = "OpalDriver")
  logindata <- builder$build()
  
  # Log onto the remote Opal training servers
  connections <- DSI::datashield.login(logins = logindata, assign = TRUE, symbol = "D") 
  
  # Fit the logistic regression model

  mod <- ds.glm(formula = "DIS_DIAB~GENDER+PM_BMI_CONTINUOUS+LAB_HDL",
                data = "D",
                family = "binomial",
                datasources = connections)
                
  mod #visualize the results of the model

# Example 3: fit a standard Gaussian linear model with an interaction
# We are using the same data as in example 2. 

mod <- ds.glm(formula = "PM_BMI_CONTINUOUS~DIS_DIAB*GENDER+LAB_HDL",
              data = "D",
              family = "gaussian",
              datasources = connections)
mod

# Clear the Datashield R sessions and logout
datashield.logout(connections) 
}

}
\author{
DataSHIELD Development Team
}
