% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{LossAversion}
\alias{LossAversion}
\title{(No) Myopic Loss Aversion in Adolescents}
\format{
A data frame with 570 observations on 7 variables:
\describe{
\item{invest}{numeric. Average proportion of tokens invested across all 9
rounds of the experiment (response variable).}
\item{gender}{factor. Gender of the player (or team of players).}
\item{male}{factor. Was (at least one of) the player(s) male (in the team)?}
\item{age}{numeric. Age of the player (or average age in case of team).}
\item{grade}{factor. School grade of the player(s).}
\item{arrangement}{factor. Investment horizon treatment with levels \code{short}
(1 round), \code{medium} (3 rounds), and \code{long} (9 rounds).}
\item{treatment}{factor. Type of treatment: long vs. short.}
}
}
\source{
Data collected by Matthias Sutter and Daniela Glätzle-Rützler, Universität
Innsbruck.
}
\usage{
LossAversion
}
\description{
Data from a behavioral economics experiment assessing the extent of myopic loss
aversion among adolescents aged 11 to 19 years. The experiment tests whether
short-term investment horizons lead to more conservative investment behavior.
}
\details{
The data were collected by Matthias Sutter and Daniela Glätzle-Rützler
(Universität Innsbruck) in an experiment with high-school students in Tyrol,
Austria (Schwaz and Innsbruck). The experiment tests the theory of myopic loss
aversion, which proposes that investors with shorter evaluation periods are more
loss-averse and thus invest less in risky assets.

Classical theory predicts that players with short investment horizons (myopic
view) should invest less due to loss aversion. However, Sutter et al. (2015)
found no evidence of myopic loss aversion in adolescents, contrary to findings
in adult populations.

The investment game structure: In each round, players could invest tokens in a
risky asset with 50\% chance of doubling or losing the investment. The treatment
varied the feedback frequency (short = every round, medium = every 3 rounds,
long = only at the end).
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(LossAversion)
# Control bounds

LossAversion$invest <- with(
  LossAversion,
  ifelse(invest <= 0, 0.000001,
    ifelse(invest >= 1, 0.999999, invest)
  )
)
# Example 1: Test for myopic loss aversion
# Do short-term players invest less? (They shouldn't, per Sutter et al.)
fit_kw <- gkwreg(
  invest ~ arrangement + age + male + grade |
    arrangement + male,
  data = LossAversion,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Effect of investment horizon (arrangement) on mean investment
#   Age and gender effects on risk-taking
# - Beta: Precision varies by horizon and gender
#   (some groups more consistent than others)

# Example 2: Interaction effects
# Does the horizon effect differ by age/grade?
fit_kw_interact <- gkwreg(
  invest ~ grade * (arrangement + age) + male |
    arrangement + male + grade,
  data = LossAversion,
  family = "kw"
)
summary(fit_kw_interact)

# Interpretation:
# - Grade × arrangement interaction tests if myopic loss aversion
#   emerges differently at different developmental stages

# Example 3: Extended-support for boundary observations
# Some students invest 0\% or 100\% of tokens
# Original 'invest' variable may include exact 0 and 1 values
fit_xbx <- gkwreg(
  invest ~ grade * (arrangement + age) + male |
    arrangement + male + grade,
  data = LossAversion,
  family = "kw" # Note: for true [0,1] support, use extended-support models
)
summary(fit_xbx)

# Interpretation:
# - Model accommodates extreme risk-taking (all-in or all-out strategies)

# Compare models
anova(fit_kw, fit_kw_interact)

# Visualization: Investment by horizon
boxplot(invest ~ arrangement,
  data = LossAversion,
  xlab = "Investment Horizon", ylab = "Proportion Invested",
  main = "No Myopic Loss Aversion in Adolescents",
  col = c("lightblue", "lightgreen", "lightyellow")
)
}
}
\references{
Sutter, M., Kocher, M.G., Glätzle-Rützler, D., and Trautmann, S.T. (2015).
No Myopic Loss Aversion in Adolescents? – An Experimental Note.
\emph{Journal of Economic Behavior & Organization}, \strong{111}, 169--176.
\doi{10.1016/j.jebo.2014.12.021}

Kosmidis, I., and Zeileis, A. (2024). Extended-Support Beta Regression for
(0, 1) Responses. \emph{arXiv:2409.07233}.
\doi{10.48550/arXiv.2409.07233}
}
\keyword{datasets}
