/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 
#define H5F_FRIEND     

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Cpkg.h"      
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 

typedef struct {
    H5F_t *f;                         
    bool   evicted_entries_last_pass; 
    bool pinned_entries_need_evicted; 
    bool skipped_pf_dirty_entries;    
} H5C_tag_iter_evict_ctx_t;

typedef struct {
    H5F_t   *f;       
    int      type_id; 
    unsigned flags;   
} H5C_tag_iter_ettm_ctx_t;

typedef struct {
    bool cork_val; 
} H5C_tag_iter_cork_ctx_t;

static herr_t H5C__iter_tagged_entries_real(H5C_t *cache, haddr_t tag, H5C_tag_iter_cb_t cb, void *cb_ctx);

H5FL_EXTERN(H5C_tag_info_t);

herr_t
H5C_ignore_tags(H5C_t *cache)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache != NULL);

    
    cache->ignore_tags = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

H5_ATTR_PURE bool
H5C_get_ignore_tags(const H5C_t *cache)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache);

    
    FUNC_LEAVE_NOAPI(cache->ignore_tags)
} 

H5_ATTR_PURE uint32_t
H5C_get_num_objs_corked(const H5C_t *cache)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache);

    
    FUNC_LEAVE_NOAPI(cache->num_objs_corked)
} 

herr_t
H5C__tag_entry(H5C_t *cache, H5C_cache_entry_t *entry)
{
    H5C_tag_info_t *tag_info;            
    haddr_t         tag;                 
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(cache != NULL);
    assert(entry != NULL);

    
    tag = H5CX_get_tag();

    if (cache->ignore_tags) {
        
        if (!H5_addr_defined(tag))
            tag = H5AC__IGNORE_TAG;
    }
#ifdef H5C_DO_TAGGING_SANITY_CHECKS
    else {
        
        if (H5C_verify_tag(entry->type->id, tag) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "tag verification failed");
    }
#endif

    
    HASH_FIND(hh, cache->tag_list, &tag, sizeof(haddr_t), tag_info);

    
    if (NULL == tag_info) {
        
        if (NULL == (tag_info = H5FL_CALLOC(H5C_tag_info_t)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "can't allocate tag info for cache entry");

        
        tag_info->tag = tag;

        
        HASH_ADD(hh, cache->tag_list, tag, sizeof(haddr_t), tag_info);
    }
    else
        assert(tag_info->corked || (tag_info->entry_cnt > 0 && tag_info->head));

    
    assert(entry->tl_next == NULL);
    assert(entry->tl_prev == NULL);
    assert(entry->tag_info == NULL);

    
    entry->tl_next  = tag_info->head;
    entry->tag_info = tag_info;
    if (tag_info->head)
        tag_info->head->tl_prev = entry;
    tag_info->head = entry;
    tag_info->entry_cnt++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__untag_entry(H5C_t *cache, H5C_cache_entry_t *entry)
{
    H5C_tag_info_t *tag_info;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cache != NULL);
    assert(entry != NULL);

    
    if (NULL != (tag_info = entry->tag_info)) {
        
        if (entry->tl_next)
            entry->tl_next->tl_prev = entry->tl_prev;
        if (entry->tl_prev)
            entry->tl_prev->tl_next = entry->tl_next;
        if (tag_info->head == entry)
            tag_info->head = entry->tl_next;
        tag_info->entry_cnt--;

        
        entry->tl_next  = NULL;
        entry->tl_prev  = NULL;
        entry->tag_info = NULL;

        
        if (!tag_info->corked && 0 == tag_info->entry_cnt) {
            
            assert(NULL == tag_info->head);

            
            HASH_DELETE(hh, cache->tag_list, tag_info);
            tag_info = H5FL_FREE(H5C_tag_info_t, tag_info);
        }
        else
            assert(tag_info->corked || NULL != tag_info->head);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__iter_tagged_entries_real(H5C_t *cache, haddr_t tag, H5C_tag_iter_cb_t cb, void *cb_ctx)
{
    H5C_tag_info_t *tag_info;            
    herr_t          ret_value = SUCCEED; 

    
    FUNC_ENTER_PACKAGE

    
    assert(cache != NULL);

    
    HASH_FIND(hh, cache->tag_list, &tag, sizeof(haddr_t), tag_info);

    
    if (tag_info) {
        H5C_cache_entry_t *entry;      
        H5C_cache_entry_t *next_entry; 

        
        assert(tag_info->head);
        assert(tag_info->entry_cnt > 0);

        
        entry = tag_info->head;
        while (entry) {
            
            next_entry = entry->tl_next;

            
            if ((cb)(entry, cb_ctx) != H5_ITER_CONT)
                HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "tagged entry iteration callback failed");

            
            entry = next_entry;
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__iter_tagged_entries(H5C_t *cache, haddr_t tag, bool match_global, H5C_tag_iter_cb_t cb, void *cb_ctx)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_PACKAGE

    
    assert(cache != NULL);

    
    if (H5C__iter_tagged_entries_real(cache, tag, cb, cb_ctx) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "iteration of tagged entries failed");

    
    if (match_global) {
        
        if (H5C__iter_tagged_entries_real(cache, H5AC__SOHM_TAG, cb, cb_ctx) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "iteration of tagged entries failed");

        
        if (H5C__iter_tagged_entries_real(cache, H5AC__GLOBALHEAP_TAG, cb, cb_ctx) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "iteration of tagged entries failed");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5C__evict_tagged_entries_cb(H5C_cache_entry_t *entry, void *_ctx)
{
    H5C_tag_iter_evict_ctx_t *ctx = (H5C_tag_iter_evict_ctx_t *)_ctx; 
    int                       ret_value = H5_ITER_CONT;               

    
    FUNC_ENTER_PACKAGE

    
    assert(entry);
    assert(ctx);

    
    if (entry->is_protected)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, H5_ITER_ERROR, "Cannot evict protected entry");
    else if (entry->is_dirty)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, H5_ITER_ERROR, "Cannot evict dirty entry");
    else if (entry->is_pinned)
        
        ctx->pinned_entries_need_evicted = true;
    else if (!entry->prefetched_dirty) {
        
        if (H5C__flush_single_entry(ctx->f, entry,
                                    H5C__FLUSH_INVALIDATE_FLAG | H5C__FLUSH_CLEAR_ONLY_FLAG |
                                        H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, H5_ITER_ERROR, "Entry eviction failed.");
        ctx->evicted_entries_last_pass = true;
    }
    else
        ctx->skipped_pf_dirty_entries = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_evict_tagged_entries(H5F_t *f, haddr_t tag, bool match_global)
{
    H5C_t                   *cache;               
    H5C_tag_iter_evict_ctx_t ctx;                 
    herr_t                   ret_value = SUCCEED; 

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache = f->shared->cache; 
    assert(cache != NULL);

    
    ctx.f = f;

    
    do {
        
        ctx.pinned_entries_need_evicted = false;
        ctx.evicted_entries_last_pass   = false;
        ctx.skipped_pf_dirty_entries    = false;

        
        if (H5C__iter_tagged_entries(cache, tag, match_global, H5C__evict_tagged_entries_cb, &ctx) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "Iteration of tagged entries failed");

        
    } while (true == ctx.evicted_entries_last_pass);

    
    if ((!ctx.skipped_pf_dirty_entries) && (ctx.pinned_entries_need_evicted))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Pinned entries still need evicted?!");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5C_DO_TAGGING_SANITY_CHECKS

herr_t
H5C_verify_tag(int id, haddr_t tag)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NOINIT

    
    if (tag == H5AC__IGNORE_TAG)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "cannot ignore a tag while doing verification.");
    else if (tag == H5AC__INVALID_TAG) {
        if (id != H5AC_PROXY_ENTRY_ID)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "no metadata tag provided");
    } 
    else {
        

        
        if ((id == H5AC_SUPERBLOCK_ID) || (id == H5AC_DRVRINFO_ID)) {
            if (tag != H5AC__SUPERBLOCK_TAG)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "superblock not tagged with H5AC__SUPERBLOCK_TAG");
        } 
        else {
            if (tag == H5AC__SUPERBLOCK_TAG)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL,
                            "H5AC__SUPERBLOCK_TAG applied to non-superblock entry");
        } 

        
        if (tag == H5AC__FREESPACE_TAG && ((id != H5AC_FSPACE_HDR_ID) && (id != H5AC_FSPACE_SINFO_ID)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "H5AC__FREESPACE_TAG applied to non-freespace entry");

        
        if ((id == H5AC_SOHM_TABLE_ID) || (id == H5AC_SOHM_LIST_ID))
            if (tag != H5AC__SOHM_TAG)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "sohm entry not tagged with H5AC__SOHM_TAG");

        
        if (id == H5AC_GHEAP_ID) {
            if (tag != H5AC__GLOBALHEAP_TAG)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "global heap not tagged with H5AC__GLOBALHEAP_TAG");
        } 
        else {
            if (tag == H5AC__GLOBALHEAP_TAG)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL,
                            "H5AC__GLOBALHEAP_TAG applied to non-globalheap entry");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif

static int
H5C__flush_tagged_entries_cb(H5C_cache_entry_t *entry, void *_ctx)
{
    H5C_t *cache_ptr = (H5C_t *)_ctx;
    int    ret_value = H5_ITER_CONT;

    
    FUNC_ENTER_PACKAGE

    
    assert(entry);
    assert(cache_ptr);

    
    if (entry->is_dirty)
        H5C__INSERT_ENTRY_IN_SLIST(cache_ptr, entry, H5_ITER_ERROR);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_flush_tagged_entries(H5F_t *f, haddr_t tag)
{
    
    H5C_t *cache     = NULL;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    cache = f->shared->cache;

    
    if (H5C_set_slist_enabled(f->shared->cache, true, false) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "set slist enabled failed");

    
    if (H5C__iter_tagged_entries(cache, tag, true, H5C__flush_tagged_entries_cb, cache) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "Iteration of tagged entries failed");

    
    if (H5C_flush_cache(f, H5C__FLUSH_IGNORE_PROTECTED_FLAG) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't flush cache");

    
    if (H5C_set_slist_enabled(f->shared->cache, false, false) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "disable slist failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_retag_entries(H5C_t *cache, haddr_t src_tag, haddr_t dest_tag)
{
    H5C_tag_info_t *tag_info = NULL;

    
    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache);

    
    HASH_FIND(hh, cache->tag_list, &src_tag, sizeof(haddr_t), tag_info);
    if (NULL != tag_info) {
        
        HASH_DELETE(hh, cache->tag_list, tag_info);

        
        tag_info->tag = dest_tag;

        
        HASH_ADD(hh, cache->tag_list, tag, sizeof(haddr_t), tag_info);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5C__expunge_tag_type_metadata_cb(H5C_cache_entry_t *entry, void *_ctx)
{
    H5C_tag_iter_ettm_ctx_t *ctx = (H5C_tag_iter_ettm_ctx_t *)_ctx; 
    int                      ret_value = H5_ITER_CONT;              

    
    FUNC_ENTER_PACKAGE

    
    assert(entry);
    assert(ctx);

    
    if (entry->type->id == ctx->type_id)
        if (H5C_expunge_entry(ctx->f, entry->type, entry->addr, ctx->flags) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, H5_ITER_ERROR, "can't expunge entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_expunge_tag_type_metadata(H5F_t *f, haddr_t tag, int type_id, unsigned flags)
{
    H5C_t                  *cache;               
    H5C_tag_iter_ettm_ctx_t ctx;                 
    herr_t                  ret_value = SUCCEED; 

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache = f->shared->cache; 
    assert(cache != NULL);

    
    ctx.f       = f;
    ctx.type_id = type_id;
    ctx.flags   = flags;

    
    if (H5C__iter_tagged_entries(cache, tag, false, H5C__expunge_tag_type_metadata_cb, &ctx) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "Iteration of tagged entries failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_tag(const void *thing, haddr_t *tag)
{
    const H5C_cache_entry_t *entry = (const H5C_cache_entry_t *)thing; 

    FUNC_ENTER_NOAPI_NOERR

    assert(entry);
    assert(entry->tag_info);
    assert(tag);

    
    *tag = entry->tag_info->tag;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
