/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FSmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FSpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 
#include "H5VMprivate.h" 

#define H5FS_HDR_VERSION   0 
#define H5FS_SINFO_VERSION 0 

typedef struct {
    H5FS_sinfo_t *sinfo;         
    uint8_t     **image;         
    unsigned      sect_cnt_size; 
} H5FS_iter_ud_t;

static herr_t H5FS__sinfo_serialize_sect_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata);
static herr_t H5FS__sinfo_serialize_node_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata);

static herr_t H5FS__cache_hdr_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5FS__cache_hdr_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5FS__cache_hdr_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5FS__cache_hdr_image_len(const void *thing, size_t *image_len);
static herr_t H5FS__cache_hdr_pre_serialize(H5F_t *f, void *thing, haddr_t addr, size_t len,
                                            haddr_t *new_addr, size_t *new_len, unsigned *flags);
static herr_t H5FS__cache_hdr_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5FS__cache_hdr_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5FS__cache_hdr_free_icr(void *thing);

static herr_t H5FS__cache_sinfo_get_initial_load_size(void *udata, size_t *image_len);
static htri_t H5FS__cache_sinfo_verify_chksum(const void *image_ptr, size_t len, void *udata_ptr);
static void  *H5FS__cache_sinfo_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5FS__cache_sinfo_image_len(const void *thing, size_t *image_len);
static herr_t H5FS__cache_sinfo_pre_serialize(H5F_t *f, void *thing, haddr_t addr, size_t len,
                                              haddr_t *new_addr, size_t *new_len, unsigned *flags);
static herr_t H5FS__cache_sinfo_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5FS__cache_sinfo_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5FS__cache_sinfo_free_icr(void *thing);

const H5AC_class_t H5AC_FSPACE_HDR[1] = {{
    H5AC_FSPACE_HDR_ID,                    
    "Free Space Header",                   
    H5FD_MEM_FSPACE_HDR,                   
    H5AC__CLASS_NO_FLAGS_SET,              
    H5FS__cache_hdr_get_initial_load_size, 
    NULL,                                  
    H5FS__cache_hdr_verify_chksum,         
    H5FS__cache_hdr_deserialize,           
    H5FS__cache_hdr_image_len,             
    H5FS__cache_hdr_pre_serialize,         
    H5FS__cache_hdr_serialize,             
    H5FS__cache_hdr_notify,                
    H5FS__cache_hdr_free_icr,              
    NULL,                                  
}};

const H5AC_class_t H5AC_FSPACE_SINFO[1] = {{
    H5AC_FSPACE_SINFO_ID,                    
    "Free Space Section Info",               
    H5FD_MEM_FSPACE_SINFO,                   
    H5AC__CLASS_NO_FLAGS_SET,                
    H5FS__cache_sinfo_get_initial_load_size, 
    NULL,                                    
    H5FS__cache_sinfo_verify_chksum,         
    H5FS__cache_sinfo_deserialize,           
    H5FS__cache_sinfo_image_len,             
    H5FS__cache_sinfo_pre_serialize,         
    H5FS__cache_sinfo_serialize,             
    H5FS__cache_sinfo_notify,                
    H5FS__cache_sinfo_free_icr,              
    NULL,                                    
}};

static herr_t
H5FS__cache_hdr_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5FS_hdr_cache_ud_t *udata = (H5FS_hdr_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(udata->f);
    assert(image_len);

    
    *image_len = (size_t)H5FS_HEADER_SIZE(udata->f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

htri_t
H5FS__cache_hdr_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;                

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FS__cache_hdr_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                            bool H5_ATTR_UNUSED *dirty)
{
    H5FS_t              *fspace = NULL;                          
    H5FS_hdr_cache_ud_t *udata  = (H5FS_hdr_cache_ud_t *)_udata; 
    const uint8_t       *image  = (const uint8_t *)_image;       
    uint32_t             stored_chksum;                          
    unsigned             nclasses;                               
    H5FS_t              *ret_value = NULL;                       

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);
    assert(udata->f);

    
    if (NULL == (fspace = H5FS__new(udata->f, udata->nclasses, udata->classes, udata->cls_init_udata)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    fspace->addr = udata->addr;

    
    if (memcmp(image, H5FS_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "wrong free space header signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5FS_HDR_VERSION)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "wrong free space header version");

    
    fspace->client = (H5FS_client_t)*image++;
    if (fspace->client >= H5FS_NUM_CLIENT_ID)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "unknown client ID in free space header");

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->tot_space);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->tot_sect_count);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->serial_sect_count);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->ghost_sect_count);

    
    
    UINT16DECODE(image, nclasses);
    if (fspace->nclasses > 0 && nclasses > fspace->nclasses)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "section class count mismatch");

    
    UINT16DECODE(image, fspace->shrink_percent);

    
    UINT16DECODE(image, fspace->expand_percent);

    
    UINT16DECODE(image, fspace->max_sect_addr);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->max_sect_size);

    
    H5F_addr_decode(udata->f, &image, &fspace->sect_addr);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->sect_size);

    
    H5F_DECODE_LENGTH(udata->f, image, fspace->alloc_sect_size);

    

    
    UINT32DECODE(image, stored_chksum);

    
    assert((size_t)(image - (const uint8_t *)_image) <= len);

    
    ret_value = fspace;

done:
    
    if (!ret_value && fspace)
        if (H5FS__hdr_dest(fspace) < 0)
            HDONE_ERROR(H5E_FSPACE, H5E_CANTFREE, NULL, "unable to destroy free space header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_hdr_image_len(const void *_thing, size_t *image_len)
{
    const H5FS_t *fspace = (const H5FS_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(fspace);
    assert(fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(image_len);

    
    *image_len = fspace->hdr_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FS__cache_hdr_pre_serialize(H5F_t *f, void *_thing, haddr_t addr, size_t H5_ATTR_UNUSED len,
                              haddr_t H5_ATTR_NDEBUG_UNUSED *new_addr, size_t H5_ATTR_NDEBUG_UNUSED *new_len,
                              unsigned *flags)
{
    H5FS_t     *fspace    = (H5FS_t *)_thing; 
    H5AC_ring_t orig_ring = H5AC_RING_INV;    
    herr_t      ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(fspace);
    assert(fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(H5_addr_defined(addr));
    assert(new_addr);
    assert(new_len);
    assert(flags);

    if (fspace->sinfo) {
        H5AC_ring_t ring;

        
        if (H5AC_get_entry_ring(f, addr, &ring) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "unable to get property value");

        
        H5AC_set_ring(ring, &orig_ring);

        
        if (fspace->serial_sect_count > 0 && H5_addr_defined(fspace->addr)) {
            
            assert(fspace->sect_size > 0);

            if (!H5_addr_defined(fspace->sect_addr)) { 
                haddr_t tag = HADDR_UNDEF;
                haddr_t sect_addr;
                hsize_t saved_sect_size, new_sect_size;

                
                saved_sect_size = fspace->sect_size;
                if (HADDR_UNDEF ==
                    (sect_addr = H5MF_alloc((H5F_t *)f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
                    HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL,
                                "file allocation failed for free space sections");

                
                if (fspace->sect_size > saved_sect_size) {

                    new_sect_size = fspace->sect_size;

                    if (H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, sect_addr, saved_sect_size) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to free free space sections");

                    if (HADDR_UNDEF ==
                        (sect_addr = H5MF_alloc((H5F_t *)f, H5FD_MEM_FSPACE_SINFO, new_sect_size)))
                        HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL,
                                    "file allocation failed for free space sections");
                    fspace->sect_size       = new_sect_size;
                    fspace->alloc_sect_size = new_sect_size;
                }
                else {
                    fspace->alloc_sect_size = saved_sect_size;
                    fspace->sect_size       = saved_sect_size;
                }
                fspace->sect_addr = sect_addr;

                
                if (H5AC_get_tag((const void *)fspace, &tag) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTTAG, FAIL, "can't get tag for metadata cache object");
                H5_BEGIN_TAG(tag)
                if (H5AC_insert_entry((H5F_t *)f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo,
                                      H5AC__NO_FLAGS_SET) < 0)
                    HGOTO_ERROR_TAG(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space sections to cache");
                H5_END_TAG

                assert(fspace->sinfo->cache_info.size == fspace->alloc_sect_size);

                
                fspace->sinfo = NULL;
            }                                                 
            else if (H5F_IS_TMP_ADDR(f, fspace->sect_addr)) { 
                haddr_t new_sect_addr;

                

                
                assert(fspace->sect_size > 0);
                assert(fspace->alloc_sect_size == (size_t)fspace->sect_size);

                
                if (HADDR_UNDEF ==
                    (new_sect_addr = H5MF_alloc((H5F_t *)f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
                    HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL,
                                "file allocation failed for free space sections");

                fspace->alloc_sect_size = (size_t)fspace->sect_size;
                assert(fspace->sinfo->cache_info.size == fspace->alloc_sect_size);

                
                if (H5AC_move_entry((H5F_t *)f, H5AC_FSPACE_SINFO, fspace->sect_addr, new_sect_addr) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTMOVE, FAIL, "unable to move section info");

                fspace->sect_addr = new_sect_addr;
            }      
            else { 
                
                assert(fspace->sect_size > 0);
                assert(fspace->alloc_sect_size == (size_t)fspace->sect_size);
            } 
        }     
        else {
            
            assert(!H5_addr_defined(fspace->sect_addr));
        } 
    }     
    else if (H5_addr_defined(fspace->sect_addr)) {
        
        if (!H5F_POINT_OF_NO_RETURN(f)) {
            assert(fspace->sect_size > 0);
            assert(fspace->alloc_sect_size == (size_t)fspace->sect_size);
        } 

        if (H5F_IS_TMP_ADDR(f, fspace->sect_addr)) {
            unsigned sect_status = 0;
            haddr_t  new_sect_addr;

            
            if (H5AC_get_entry_status(f, fspace->sect_addr, &sect_status) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get section info status");

            assert(sect_status & H5AC_ES__IN_CACHE);
            assert((sect_status & H5AC_ES__IS_PROTECTED) == 0);
            assert((sect_status & H5AC_ES__IS_PINNED) == 0);

            
            if (HADDR_UNDEF ==
                (new_sect_addr = H5MF_alloc((H5F_t *)f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
                HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL, "file allocation failed for free space sections");

            fspace->alloc_sect_size = (size_t)fspace->sect_size;

            
            assert(!H5_addr_eq(fspace->sect_addr, new_sect_addr));

            
            if (H5AC_move_entry((H5F_t *)f, H5AC_FSPACE_SINFO, fspace->sect_addr, new_sect_addr) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTMOVE, FAIL, "unable to move section info");

            
            fspace->sect_addr = new_sect_addr;

            
        }  
    }      
    else { 
        
        assert(fspace->serial_sect_count == 0);
        assert(fspace->tot_sect_count == fspace->ghost_sect_count);
    } 

    
    *flags = 0;

done:
    
    if (orig_ring != H5AC_RING_INV)
        H5AC_set_ring(orig_ring, NULL);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_hdr_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_thing)
{
    H5FS_t  *fspace = (H5FS_t *)_thing;  
    uint8_t *image  = (uint8_t *)_image; 
    uint32_t metadata_chksum;            
    herr_t   ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(image);
    assert(fspace);
    assert(fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(fspace->hdr_size == len);

    
    assert((!H5_addr_defined(fspace->sect_addr)) || (!H5F_IS_TMP_ADDR(f, fspace->sect_addr)));

    if (!H5F_POINT_OF_NO_RETURN(f))
        assert((!H5_addr_defined(fspace->sect_addr)) ||
               ((fspace->sect_size > 0) && (fspace->alloc_sect_size == (size_t)fspace->sect_size)));

    
    H5MM_memcpy(image, H5FS_HDR_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5FS_HDR_VERSION;

    
    H5_CHECKED_ASSIGN(*image++, uint8_t, fspace->client, int);

    
    H5F_ENCODE_LENGTH(f, image, fspace->tot_space);

    
    H5F_ENCODE_LENGTH(f, image, fspace->tot_sect_count);

    
    H5F_ENCODE_LENGTH(f, image, fspace->serial_sect_count);

    
    H5F_ENCODE_LENGTH(f, image, fspace->ghost_sect_count);

    
    UINT16ENCODE(image, fspace->nclasses);

    
    UINT16ENCODE(image, fspace->shrink_percent);

    
    UINT16ENCODE(image, fspace->expand_percent);

    
    UINT16ENCODE(image, fspace->max_sect_addr);

    
    H5F_ENCODE_LENGTH(f, image, fspace->max_sect_size);

    
    H5F_addr_encode(f, &image, fspace->sect_addr);

    
    H5F_ENCODE_LENGTH(f, image, fspace->sect_size);

    
    H5F_ENCODE_LENGTH(f, image, fspace->alloc_sect_size);

    
    metadata_chksum = H5_checksum_metadata((uint8_t *)_image, (size_t)(image - (uint8_t *)_image), 0);

    
    UINT32ENCODE(image, metadata_chksum);

    
    assert((size_t)(image - (uint8_t *)_image) == fspace->hdr_size);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS__cache_hdr_notify(H5AC_notify_action_t action, void *_thing)
{
    H5FS_t *fspace    = (H5FS_t *)_thing; 
    herr_t  ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(fspace);

    
    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            if (H5AC_unsettle_entry_ring(fspace) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTFLUSH, FAIL, "unable to mark FSM ring as unsettled");
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_FSPACE, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_hdr_free_icr(void *_thing)
{
    H5FS_t *fspace    = (H5FS_t *)_thing; 
    herr_t  ret_value = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    assert(fspace);
    assert(fspace->cache_info.type == H5AC_FSPACE_HDR);

    
    assert(!fspace->sinfo);

    
    if (H5FS__hdr_dest(fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to destroy free space header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_sinfo_get_initial_load_size(void *_udata, size_t *image_len)
{
    const H5FS_t          *fspace;                                  
    H5FS_sinfo_cache_ud_t *udata = (H5FS_sinfo_cache_ud_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    fspace = udata->fspace;
    assert(fspace);
    assert(fspace->sect_size > 0);
    assert(image_len);

    
    *image_len = (size_t)(fspace->sect_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

htri_t
H5FS__cache_sinfo_verify_chksum(const void *_image, size_t len, void H5_ATTR_UNUSED *_udata)
{
    const uint8_t *image = (const uint8_t *)_image; 
    uint32_t       stored_chksum;                   
    uint32_t       computed_chksum;                 
    htri_t         ret_value = true;                

    FUNC_ENTER_PACKAGE

    
    assert(image);

    
    if (H5F_get_checksums(image, len, &stored_chksum, &computed_chksum) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL, "can't get checksums");

    if (stored_chksum != computed_chksum)
        ret_value = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FS__cache_sinfo_deserialize(const void *_image, size_t H5_ATTR_NDEBUG_UNUSED len, void *_udata,
                              bool H5_ATTR_NDEBUG_UNUSED *dirty)
{
    H5FS_sinfo_cache_ud_t *udata = (H5FS_sinfo_cache_ud_t *)_udata; 
    H5FS_t                *fspace;                                  
    H5FS_sinfo_t          *sinfo = NULL;                            
    haddr_t                fs_addr;                                 
    size_t                 old_sect_size;                           
    const uint8_t         *image = (const uint8_t *)_image;         
    const uint8_t         *chksum_image;                            
    uint32_t               stored_chksum;                           
    void                  *ret_value = NULL;                        

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);
    fspace = udata->fspace;
    assert(fspace);
    assert(fspace->sect_size == len);
    assert(dirty);

    
    if (NULL == (sinfo = H5FS__sinfo_new(udata->f, fspace)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    H5_CHECKED_ASSIGN(old_sect_size, size_t, fspace->sect_size, hsize_t);

    
    if (memcmp(image, H5FS_SINFO_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "wrong free space sections signature");
    image += H5_SIZEOF_MAGIC;

    
    if (*image++ != H5FS_SINFO_VERSION)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "wrong free space sections version");

    
    H5F_addr_decode(udata->f, &image, &fs_addr);
    if (H5_addr_ne(fs_addr, fspace->addr))
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTLOAD, NULL, "incorrect header address for free space sections");

    
    if (fspace->serial_sect_count > 0) {
        hsize_t old_tot_sect_count; 
        hsize_t H5_ATTR_NDEBUG_UNUSED
            old_serial_sect_count;                          
        hsize_t H5_ATTR_NDEBUG_UNUSED old_ghost_sect_count; 
        hsize_t H5_ATTR_NDEBUG_UNUSED old_tot_space;        
        unsigned                      sect_cnt_size;        

        
        sect_cnt_size = H5VM_limit_enc_size((uint64_t)fspace->serial_sect_count);

        
        old_tot_sect_count        = fspace->tot_sect_count;
        old_serial_sect_count     = fspace->serial_sect_count;
        old_ghost_sect_count      = fspace->ghost_sect_count;
        old_tot_space             = fspace->tot_space;
        fspace->tot_sect_count    = 0;
        fspace->serial_sect_count = 0;
        fspace->ghost_sect_count  = 0;
        fspace->tot_space         = 0;

        
        do {
            hsize_t sect_size  = 0; 
            size_t  node_count = 0; 
            size_t  u;              

            
            UINT64DECODE_VAR(image, node_count, sect_cnt_size);
            assert(node_count);

            
            UINT64DECODE_VAR(image, sect_size, sinfo->sect_len_size);
            assert(sect_size);

            
            for (u = 0; u < node_count; u++) {
                H5FS_section_info_t *new_sect;      
                haddr_t              sect_addr = 0; 
                unsigned             sect_type;     
                unsigned             des_flags;     

                
                UINT64DECODE_VAR(image, sect_addr, sinfo->sect_off_size);

                
                sect_type = *image++;

                
                des_flags = 0;
                assert(fspace->sect_cls[sect_type].deserialize);
                if (NULL == (new_sect = (*fspace->sect_cls[sect_type].deserialize)(
                                 &fspace->sect_cls[sect_type], image, sect_addr, sect_size, &des_flags)))
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTDECODE, NULL, "can't deserialize section");

                
                image += fspace->sect_cls[sect_type].serial_size;

                
                if (!(des_flags & H5FS_DESERIALIZE_NO_ADD))
                    if (H5FS_sect_add(udata->f, fspace, new_sect, H5FS_ADD_DESERIALIZING, udata) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINSERT, NULL,
                                    "can't add section to free space manager");
            } 

            if (fspace->tot_sect_count == old_tot_sect_count)
                break;

        } while (image < (((const uint8_t *)_image + old_sect_size) - H5FS_SIZEOF_CHKSUM));

        
        assert((size_t)(image - (const uint8_t *)_image) <= (old_sect_size - H5FS_SIZEOF_CHKSUM));
        assert(old_sect_size == fspace->sect_size);
        assert(old_tot_sect_count == fspace->tot_sect_count);
        assert(old_serial_sect_count == fspace->serial_sect_count);
        assert(old_ghost_sect_count == fspace->ghost_sect_count);
        assert(old_tot_space == fspace->tot_space);
    } 

    

    
    chksum_image = (const uint8_t *)(_image) + old_sect_size - H5FS_SIZEOF_CHKSUM;
    
    UINT32DECODE(chksum_image, stored_chksum);

    
    assert((image == chksum_image) ||
           ((size_t)((image - (const uint8_t *)_image) + (chksum_image - image)) == old_sect_size));

    
    ret_value = sinfo;

done:
    if (!ret_value && sinfo)
        if (H5FS__sinfo_dest(sinfo) < 0)
            HDONE_ERROR(H5E_FSPACE, H5E_CANTFREE, NULL, "unable to destroy free space info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_sinfo_image_len(const void *_thing, size_t *image_len)
{
    const H5FS_sinfo_t *sinfo = (const H5FS_sinfo_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sinfo);
    assert(sinfo->cache_info.type == H5AC_FSPACE_SINFO);
    assert(sinfo->fspace);
    assert(sinfo->fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(image_len);

    
    H5_CHECKED_ASSIGN(*image_len, size_t, sinfo->fspace->alloc_sect_size, hsize_t);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FS__cache_sinfo_pre_serialize(H5F_t *f, void *_thing, haddr_t addr, size_t H5_ATTR_NDEBUG_UNUSED len,
                                haddr_t *new_addr, size_t H5_ATTR_NDEBUG_UNUSED *new_len, unsigned *flags)
{
    H5FS_sinfo_t *sinfo = (H5FS_sinfo_t *)_thing; 
    H5FS_t       *fspace;                         
    haddr_t       sinfo_addr;                     
    herr_t        ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(sinfo);
    assert(sinfo->cache_info.type == H5AC_FSPACE_SINFO);
    fspace = sinfo->fspace;
    assert(fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(fspace->cache_info.is_pinned);
    assert(H5_addr_defined(addr));
    assert(H5_addr_eq(fspace->sect_addr, addr));
    assert(fspace->sect_size == len);
    assert(new_addr);
    assert(new_len);
    assert(flags);

    sinfo_addr = addr; 

    
    if (H5F_IS_TMP_ADDR(f, fspace->sect_addr)) {
        
        assert(fspace->sect_size > 0);
        assert(H5_addr_eq(fspace->sect_addr, addr));

        
        if (HADDR_UNDEF == (sinfo_addr = H5MF_alloc((H5F_t *)f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
            HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL, "file allocation failed for free space sections");

        fspace->alloc_sect_size = (size_t)fspace->sect_size;

        
        assert(!H5_addr_eq(sinfo->fspace->sect_addr, sinfo_addr));

        
        if (H5AC_move_entry((H5F_t *)f, H5AC_FSPACE_SINFO, sinfo->fspace->sect_addr, sinfo_addr) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMOVE, FAIL, "unable to move section info");

        
        sinfo->fspace->sect_addr = sinfo_addr;

        
        if (H5AC_mark_entry_dirty(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL, "unable to mark free space header as dirty");
    } 

    if (!H5_addr_eq(addr, sinfo_addr)) {
        *new_addr = sinfo_addr;
        *flags    = H5C__SERIALIZE_MOVED_FLAG;
    } 
    else
        *flags = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_sinfo_serialize(const H5F_t *f, void *_image, size_t len, void *_thing)
{
    H5FS_sinfo_t  *sinfo = (H5FS_sinfo_t *)_thing; 
    H5FS_iter_ud_t udata;                          
    ptrdiff_t      gap_size;
    uint8_t       *image        = (uint8_t *)_image; 
    uint8_t       *chksum_image = NULL;              
    uint32_t       metadata_chksum;                  
    unsigned       bin;                              
    herr_t         ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(image);
    assert(sinfo);
    assert(sinfo->cache_info.type == H5AC_FSPACE_SINFO);
    assert(sinfo->fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(sinfo->fspace->cache_info.is_pinned);
    assert(sinfo->fspace->sect_size == len);
    assert(sinfo->fspace->sect_cls);

    
    H5MM_memcpy(image, H5FS_SINFO_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5FS_SINFO_VERSION;

    
    H5F_addr_encode(f, &image, sinfo->fspace->addr);

    
    udata.sinfo         = sinfo;
    udata.image         = &image;
    udata.sect_cnt_size = H5VM_limit_enc_size((uint64_t)sinfo->fspace->serial_sect_count);

    
    for (bin = 0; bin < sinfo->nbins; bin++)
        
        if (sinfo->bins[bin].bin_list)
            
            if (H5SL_iterate(sinfo->bins[bin].bin_list, H5FS__sinfo_serialize_node_cb, &udata) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't iterate over section size nodes");

    

    
    chksum_image = (uint8_t *)(_image) + len - H5FS_SIZEOF_CHKSUM;

    
    gap_size = chksum_image - image;
    if (gap_size > 0)
        memset(image, 0, (size_t)gap_size);

    metadata_chksum = H5_checksum_metadata(_image, (size_t)(chksum_image - (uint8_t *)_image), 0);
    
    UINT32ENCODE(chksum_image, metadata_chksum);

    
    assert((chksum_image == image) ||
           ((size_t)((image - (uint8_t *)_image) + (chksum_image - image)) == sinfo->fspace->sect_size));
    assert(sinfo->fspace->sect_size <= sinfo->fspace->alloc_sect_size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS__cache_sinfo_notify(H5AC_notify_action_t action, void *_thing)
{
    H5FS_sinfo_t *sinfo     = (H5FS_sinfo_t *)_thing;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);

    
    if (sinfo->fspace->swmr_write) {
        
        switch (action) {
            case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            case H5AC_NOTIFY_ACTION_AFTER_LOAD:
                
                if (H5FS__create_flush_depend((H5AC_info_t *)sinfo->fspace, (H5AC_info_t *)sinfo) < 0)
                    HGOTO_ERROR(
                        H5E_FSPACE, H5E_CANTDEPEND, FAIL,
                        "unable to create flush dependency between data block and header, address = %llu",
                        (unsigned long long)sinfo->fspace->sect_addr);
                break;

            case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
            case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
                
                break;

            case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
                
                if (H5FS__destroy_flush_depend((H5AC_info_t *)sinfo->fspace, (H5AC_info_t *)sinfo) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
                break;

            default:
#ifdef NDEBUG
                HGOTO_ERROR(H5E_FSPACE, H5E_BADVALUE, FAIL, "unknown action from metadata cache");
#else     
                assert(0 && "Unknown action?!?");
#endif    
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__cache_sinfo_free_icr(void *_thing)
{
    H5FS_sinfo_t *sinfo     = (H5FS_sinfo_t *)_thing; 
    herr_t        ret_value = SUCCEED;                

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);
    assert(sinfo->cache_info.type == H5AC_FSPACE_SINFO);
    assert(sinfo->fspace->cache_info.type == H5AC_FSPACE_HDR);
    assert(sinfo->fspace->cache_info.is_pinned);

    
    if (H5FS__sinfo_dest(sinfo) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to destroy free space info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sinfo_serialize_sect_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata)
{
    H5FS_section_class_t *sect_cls;                                 
    H5FS_section_info_t  *sect      = (H5FS_section_info_t *)_item; 
    H5FS_iter_ud_t       *udata     = (H5FS_iter_ud_t *)_udata;     
    herr_t                ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(udata->sinfo);
    assert(udata->image);

    
    sect_cls = &udata->sinfo->fspace->sect_cls[sect->type];

    
    if (!(sect_cls->flags & H5FS_CLS_GHOST_OBJ)) {
        
        UINT64ENCODE_VAR(*udata->image, sect->addr, udata->sinfo->sect_off_size);

        
        *(*udata->image)++ = (uint8_t)sect->type;

        
        if (sect_cls->serialize) {
            if ((*sect_cls->serialize)(sect_cls, sect, *udata->image) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTSERIALIZE, FAIL, "can't synchronize section");

            
            (*udata->image) += sect_cls->serial_size;
        } 
        else
            assert(sect_cls->serial_size == 0);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sinfo_serialize_node_cb(void *_item, void H5_ATTR_UNUSED *key, void *_udata)
{
    H5FS_node_t    *fspace_node = (H5FS_node_t *)_item;     
    H5FS_iter_ud_t *udata       = (H5FS_iter_ud_t *)_udata; 
    herr_t          ret_value   = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    assert(fspace_node);
    assert(udata->sinfo);
    assert(udata->image);

    
    if (fspace_node->serial_count > 0) {
        
        UINT64ENCODE_VAR(*udata->image, fspace_node->serial_count, udata->sect_cnt_size);

        
        UINT64ENCODE_VAR(*udata->image, fspace_node->sect_size, udata->sinfo->sect_len_size);

        
        assert(fspace_node->sect_list);
        if (H5SL_iterate(fspace_node->sect_list, H5FS__sinfo_serialize_sect_cb, udata) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_BADITER, FAIL, "can't iterate over section nodes");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
