% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter.R
\name{filter_row}
\alias{filter_row}
\title{Subset rows using annotation values}
\usage{
filter_row(.ms, ..., .preserve = FALSE)
}
\arguments{
\item{.ms}{\code{matrixset} object to subset based on the filtering
conditions}

\item{...}{Condition, or expression, that returns a logical value,
used to determine if rows are kept or discarded. The
expression may refer to row annotations - columns of
the \code{row_info} component of \code{.ms} More than one
condition can be supplied and if multiple
expressions are included, they are combined with the \code{&}
operator. Only rows for which all conditions evaluate to
TRUE are kept.}

\item{.preserve}{\code{logical}, relevant only if \code{.ms} is row grouped. When
\code{.preserve} is \code{FALSE} (the default), the row grouping
is updated based on the new \code{matrixset} resulting from
the filtering. Otherwise, the row grouping is kept as is.}
}
\value{
A \code{matrixset}, with possibly a subset of the rows of the original object.
Groups will be updated if \code{.preserve} is \code{TRUE}.
}
\description{
The \code{\link[=filter_row]{filter_row()}} function subsets the rows of all matrices of a
\code{matrixset}, retaining all rows that satisfy given condition(s). The function
\code{filter_row} works like \code{dplyr}'s \code{\link[dplyr:filter]{dplyr::filter()}}.
}
\details{
The conditions are given as expressions in \code{...}, which are applied to
columns of the annotation data frame (\code{row_info}) to determine which rows
should be retained.

It can be applied to both grouped and ungrouped \code{matrixset} (see
\code{\link[=row_group_by]{row_group_by()}}), and section \sQuote{Grouped matrixsets}.
}
\section{Grouped matrixsets}{

Column grouping (\code{\link[=column_group_by]{column_group_by()}}) has no impact on row filtering.

The impact of row grouping (\code{\link[=row_group_by]{row_group_by()}}) on row filtering depends on
the conditions. Often, row grouping will not have any impact, but as soon as
an aggregating, lagging or ranking function is involved, then the results
will differ.

For instance, the two following are not equivalent (except by pure
coincidence).

\code{student_results \%>\% filter_row(previous_year_score > mean(previous_year_score))}

And it's grouped equivalent:
\code{student_results \%>\% row_group_by(class) \%>\% filter_row(previous_year_score > mean(previous_year_score))}

In the ungrouped version, the mean of \code{previous_year_score} is taken globally
and \code{filter_row} keeps rows with \code{previous_year_score} greater than this
global average. In the grouped version, the average is calculated within each
\code{class} and the kept rows are the ones with \code{previous_year_score} greater
than the within-class average.
}

\examples{
# Filtering using one condition
filter_row(student_results, class == "classA")

# Filetring using multiple conditions. These are equivalent
filter_row(student_results, class == "classA" & previous_year_score > 0.75)
filter_row(student_results, class == "classA", previous_year_score > 0.75)

# The potential difference between grouped and non-grouped.
filter_row(student_results, previous_year_score > mean(previous_year_score))
sr <- row_group_by(student_results, teacher)
filter_row(sr, previous_year_score > mean(previous_year_score))

}
