#' Fits the McCullagh (1978) conditional-symmetry model.
#'
#' McCullagh, P. (1978).  A class of parametric models for the analysis of
#' square contingency tables with ordered categories.  Biometrika, 65(2) 413-418.
#' @param n matrix of observed counts
#' @param max_iter maximum number of iterations to maximize the log(likelihood)
#' @param verbose should cycle-by-cycle info be printed. Default is FALSE.
#' @returns a list containing
#'    theta: the asymmetry parameter
#'    chisq: chi-square
#'    g_squared: likelihood ratio G^2
#'    df: degrees of freedom
#' @export
#' @examples
#' McCullagh_conditional_symmetry(vision_data)
McCullagh_conditional_symmetry <- function(n, max_iter=5, verbose=FALSE) {
  r <- nrow(n)
  phi <- McCullagh_conditional_symmetry_initialize_phi(nrow(n))
  theta <- 1.0
  pi <- McCullagh_conditional_symmetry_pi(phi, theta)
  if (verbose) {
    message(paste("0 : ", log_likelihood(n, pi), "\n"))
  }
  for (iter in 1:max_iter) {
    phi <- McCullagh_conditional_symmetry_maximize_phi(n)
    theta <- McCullagh_conditional_symmetry_maximize_theta(n)
    pi <- McCullagh_conditional_symmetry_pi(phi, theta)
    if (verbose) {
      message(paste(iter, ": ", log_likelihood(n, pi), "\n"))
    }
  }
  if (verbose) {
    message(paste("theta = ", theta, "\n"))
  }
  df <- (r - 1) * (r - 2) / 2 + r - 2
  chisq <- pearson_chisq(n, pi)
  g_squared <- likelihood_ratio_chisq(n, pi)
  list(theta=theta, chisq=chisq, g_squared=g_squared, df=df)
}


#' Initializes symmetry matrix phi
#'
#' @param M the number of rows/columns in phi
#' @returns the phi matrix
McCullagh_conditional_symmetry_initialize_phi <- function(M) {
  phi <- matrix(0.2, nrow=M, ncol=M)
  for (i in 1:M) {
    phi[i, i] <- 1.0
  }
  const <- sum(phi)
  phi <- phi / const
}


#' Computes model-based proportions.
#'
#' @param phi the symmetric matrix
#' @param theta the asymmetry parameter
#' @returns matrix of model-based p-values
McCullagh_conditional_symmetry_pi <- function(phi, theta) {
  M <- nrow(phi)
  c <- 0.0
  pi <- matrix(0.0, nrow=nrow(phi), ncol=ncol(phi))

  for (i in 1:M) {
    for (j in 1:M) {
      if (j == i) {
        pi[i, j] <- phi[i, j]
      } else if (i < j) {
        pi[i, j] <- theta * phi[i, j]
      } else {
        pi[i, j] <- (2.0 - theta) * phi[i, j]
      }
    }
  }
  const <- sum(pi)
  pi <- pi / const
}


#' Maximizes log(likelihood) wrt phi.
#'
#' @param n matrix of observed counts
#' @returns phi matrix
McCullagh_conditional_symmetry_maximize_phi <- function(n) {
  M <- nrow(n)
  N <- sum(n)
  phi <- matrix(0.0, nrow=M, ncol=M)

  for (i in 1:M) {
    for (j in 1:i) {
      if (j == i) {
        phi[i, i] <- n[i, i] / N
      } else {
        phi[i, j] <- (n[i, j] + n[j, i]) / (2.0 * N)
      }
      phi[j, i] <- phi[i, j]
    }
  }
  phi
}


#' Computes sums used in maximizing theta.
#'
#' @param n matrix of observed counts
#' @returns list with s_i_plus and s_plus-i
McCullagh_conditional_symmetry_compute_s <- function(n) {
  M <- nrow(n)
  s_i_plus <- 0.0
  s_plus_i <- 0.0
  for (i in 1:M) {
    for (j in 1:M) {
      if (i < j) {
        s_i_plus <- s_i_plus + n[i, j]
      } else if (j < i) {
        s_plus_i = s_plus_i + n[i, j]
      }
    }
  }
  list("s_i_plus"=s_i_plus, "s_plus_i"=s_plus_i)
}


#' Maximizes the log(likelihood) wrt theta.
#'
#' @param n matrix of observed counts
#' @returns value of asymmetry parameter theta
McCullagh_conditional_symmetry_maximize_theta = function(n) {
  M <- nrow(n)
  result <- McCullagh_conditional_symmetry_compute_s(n)
  s_i_plus <- result$s_i_plus
  s_plus_i <- result$s_plus_i
  theta <- 2.0 * s_i_plus / (s_i_plus + s_plus_i)
}
