#' @title One-Compartment IV Bolus Pharmacokinetic Model (Linear)
#' @name one_compartment_iv_bolus
#' @description
#' Fits plasma concentration-time data to the one-compartment intravenous (IV) bolus
#' pharmacokinetic model. The model assumes instantaneous drug distribution throughout
#' a single, well-mixed compartment and first-order elimination kinetics.
#'
#' The function performs linear regression on log-transformed plasma concentration
#' versus time to estimate the elimination rate constant (k_el), elimination half-life
#' (t1/2), initial concentration (C0), apparent volume of distribution (Vd), and
#' clearance (CL). Optional grouping (e.g., formulation, subject) and pH-dependent
#' analysis are supported. Publication-quality plots with fitted regression lines
#' and parameter annotations are generated.
#'
#' Model:
#' C(t) = C0 * exp(-k_el * t)
#'
#' Linearized form:
#' log(C) = log(C0) - k_el * t
#'
#' where:
#' \itemize{
#'   \item C(t) is plasma concentration at time t
#'   \item C0 is the initial plasma concentration
#'   \item k_el is the elimination rate constant
#' }
#'
#' Pharmacokinetic parameters:
#' \itemize{
#'   \item Elimination half-life: t1/2 = ln(2) / k_el
#'   \item Clearance: CL = Dose / AUC
#'   \item Volume of distribution: Vd = CL / k_el
#' }
#'
#' @param data A data frame containing plasma concentration-time data.
#' @param time_col Character string specifying the column name for time.
#' @param conc_col Character string specifying the column name for plasma concentration.
#' @param dose Numeric value specifying the administered IV bolus dose.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, subject).
#' @param plot Logical; if TRUE, generates a concentration-time plot with fitted lines.
#' @param annotate Logical; if TRUE, annotates the plot with PK parameters
#'   (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text
#' scale_y_log10 labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with C0, k_el, t1/2, Vd, CL, and R2.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject IV bolus data
#' df <- data.frame(
#'   time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(18.2, 16.1, 13.5, 10.2, 6.8, 4.9, 3.6, 2.1)
#' )
#' one_compartment_iv_bolus(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent pharmacokinetics (e.g., pH or physiological state)
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6), 2),
#'   concentration = c(
#'     17.8, 15.6, 13.1, 9.8, 6.4, 4.8,   # Condition A
#'     14.9, 13.0, 10.9, 8.0, 5.2, 3.9    # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 6)
#' )
#' one_compartment_iv_bolus(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects (population-style IV bolus pharmacokinetics)
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 6),
#'   concentration = c(
#'     18.6, 16.3, 13.9, 10.5, 7.0, 5.1, 3.8,   # Subject 1
#'     17.9, 15.7, 13.2, 9.9, 6.6, 4.9, 3.6,    # Subject 2
#'     17.1, 15.0, 12.6, 9.4, 6.3, 4.7, 3.4,    # Subject 3
#'     16.4, 14.4, 12.0, 9.0, 6.0, 4.4, 3.2,    # Subject 4
#'     15.8, 13.9, 11.6, 8.7, 5.8, 4.2, 3.1,    # Subject 5
#'     15.2, 13.3, 11.0, 8.3, 5.5, 4.0, 2.9     # Subject 6
#'   ),
#'   subject = rep(paste0("S", 1:6), each = 7)
#' )
#' one_compartment_iv_bolus(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Widmark, E. M. P. (1919) Studies in the concentration of indifferent
#' narcotics in blood and tissues. Acta Medica Scandinavica, 52(1), 87–164.
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
#' Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "log_conc", "C0", "kel", "t_half", "Vd",
                         "CL", "R2", "label", "x_pos", "y_pos", "hjust", "vjust",
                         "head", "tail"))

one_compartment_iv_bolus <- function(data,
                                     time_col = "time",
                                     conc_col = "concentration",
                                     dose,
                                     group_col = NULL,
                                     plot = TRUE,
                                     annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  # Remove non-positive concentrations (log undefined)
  df <- df[df$time >= 0 & df$conc > 0, ]

  df$log_conc <- log(df$conc)

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }
  df$group <- as.factor(df$group)

  # -------------------------
  # Model fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::lm(log_conc ~ time, data = d)
    s <- summary(fit)

    kel <- -coef(fit)[2]
    logC0 <- coef(fit)[1]
    C0 <- exp(logC0)

    t_half <- log(2) / kel

    # AUC using trapezoidal rule
    auc <- sum(diff(d$time) * (head(d$conc, -1) + tail(d$conc, -1)) / 2)

    CL <- dose / auc
    Vd <- CL / kel

    data.frame(
      group = unique(d$group),
      C0 = C0,
      kel = kel,
      t_half = t_half,
      Vd = Vd,
      CL = CL,
      R2 = s$r.squared
    )
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(
      df,
      ggplot2::aes(x = time, y = conc, color = group)
    ) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(
        ggplot2::aes(y = exp(log_conc)),
        method = "lm",
        se = FALSE,
        color = "black",
        linewidth = 1
      ) +
      ggplot2::scale_y_log10() +
      ggplot2::labs(
        title = "One-Compartment IV Bolus Pharmacokinetic Model",
        subtitle = "Log-linear elimination phase",
        x = "Time (hours)",
        y = "Plasma Concentration (log scale)",
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Annotations
    # -------------------------
    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {

      ann <- fit_results
      ann$label <- paste0(
        "C0 = ", round(ann$C0, 2), "\n",
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "R2 = ", round(ann$R2, 3)
      )

      x_min <- min(df$time)
      x_max <- max(df$time)
      y_min <- min(df$conc)
      y_max <- max(df$conc)

      ann$x_pos <- c(
        x_min + 0.05 * (x_max - x_min),
        x_max - 0.05 * (x_max - x_min)
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_min * 0.5,
        y_max / 0.5
      )[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(0, 1)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
          hjust = ann$hjust,
          vjust = ann$vjust,
          size = 4,
          show.legend = FALSE
        )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
