% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/higuchi_model.R
\name{higuchi_release}
\alias{higuchi_release}
\title{Higuchi Drug Release Kinetic Model}
\arguments{
\item{data}{A data frame containing experimental drug release data.}

\item{sqrt_time_col}{Character string specifying the column name for square
root of time.}

\item{release_col}{Character string specifying the column name for
cumulative percent drug released.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, batch). Default is NULL.}

\item{pH_col}{Optional character string specifying a column containing pH
values. If provided, Higuchi models are fitted separately for each pH.}

\item{plot}{Logical; if TRUE, generates a plot (default = TRUE).}

\item{annotate}{Logical; if TRUE, annotates the plot with kH, intercept, R^2,
and t50 values (only if <= 2 groups).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{A data frame with kH, intercept, R^2, and t50
        values for each group or pH condition.}
  \item{\code{data}}{The processed data used for fitting and plotting.}
}
}
\description{
Fits experimental cumulative drug release data to the Higuchi square-root
kinetic model using linear regression of cumulative percent drug released versus
square root of time. The function supports optional grouping variables
(e.g., formulation, batch) and optional pH-dependent analysis. It generates
publication-quality plots with experimental curves, fitted Higuchi straight
lines, and annotations for Higuchi release constant (kH), intercept,
coefficient of determination (R^2), and the time required for 50-percent drug
release (t50).
}
\examples{
# Example I: Single Formulation
df <- data.frame(
  sqrt_time = c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
  release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
)
higuchi_release(
  data = df,
  sqrt_time_col = "sqrt_time",
  release_col = "release"
)

# Example II: Two formulations (grouped, not pH-dependent)
df_2 <- data.frame(
  sqrt_time = c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
  release = c(
    0, 11.4, 20.8, 30.8, 39.8, 57.8, 72.0, 84.8, 93.5,   # Formulation A
    0, 10.2, 18.6, 29.7, 37.8, 56.5, 71.9, 83.7, 92.9   # Formulation B
  ),
  formulation = rep(c("Formulation A", "Formulation B"), each = 9)
)
higuchi_release(
  data = df_2,
  sqrt_time_col = "sqrt_time",
  release_col = "release",
  group_col = "formulation"
)

# Example III: pH-dependent Higuchi release
df_pH <- data.frame(
  sqrt_time = rep(
    c(0, 3.873, 5.477, 6.708, 7.746, 9.487, 10.954, 12.247, 13.416),
    2
  ),
  release = c(
    0, 11.4, 20.8, 30.8, 39.8, 57.8, 72.0, 84.8, 93.5,  # pH 7.4
    0, 17.2, 23.8, 35.5, 41.5, 58.3, 73.6, 86.2, 93.1   # pH 4.5
  ),
  pH = rep(c(7.4, 4.5), each = 9)
)
higuchi_release(
  data = df_pH,
  sqrt_time_col = "sqrt_time",
  release_col = "release",
  pH_col = "pH"
)

# Example IV: Two formulations under two pH conditions
df1 <- data.frame(
  sqrt_time = rep(c(0.0, 2.5, 4.0, 5.2, 6.3, 7.4, 8.6, 9.8, 11.0, 12.2), 2),
  release = c(
    0.0, 12.5, 21.8, 31.2, 39.6, 50.8, 63.5, 74.2, 84.9, 92.8,  # pH 7.4
    0.0, 9.8, 18.6, 26.9, 34.7, 45.3, 56.8, 67.9, 77.4, 85.2   # pH 4.5
  ),
  pH = rep(c(7.4, 4.5), each = 10)
)
df2 <- data.frame(
  sqrt_time = rep(c(0.0, 2.5, 4.0, 5.2, 6.3, 7.4, 8.6, 9.8, 11.0, 12.2), 2),
  release = c(
    0.0, 11.3, 20.4, 29.1, 37.8, 48.6, 60.1, 71.0, 81.5, 89.6,  # pH 7.4
    0.0, 8.9, 16.7, 24.6, 32.1, 42.5, 53.4, 64.0, 73.1, 80.8   # pH 4.5
  ),
  pH = rep(c(7.4, 4.5), each = 10)
)
df_all <- rbind(
  cbind(formulation = "Dataset 1", df1),
  cbind(formulation = "Dataset 2", df2)
)
higuchi_release(
  data = df_all,
  sqrt_time_col = "sqrt_time",
  release_col = "release",
  group_col = "formulation",
  pH_col = "pH"
)
}
\references{
Higuchi, T. (1963) <doi:10.1002/jps.2600521210> Mechanism of
sustained-action medication. Journal of Pharmaceutical Sciences, 52(12),
1145–1149.
}
\author{
Paul Angelo C. Manlapaz
}
