#' @title
#' Computing upsets.
#'
#' @description
#' Computes the upset of a set of elements of the input poset.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#' @param elements a vector of character strings (the names of the input elements).
#'
#' @return
#' A vector of character strings (the names of the poset elements in the upset).
#'
#' @examples
#' elems<- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' up <- UpsetOf(pos, c("a","c"))
#'
#' @name UpsetOf
#' @export UpsetOf
UpsetOf <- function(poset, elements) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    if (!is.character(elements))
      stop("element must be a string vector")
    elements <- unique(elements)
    result <- .Call("_UpsetOf", poset@ptr, elements)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
