% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatially_balanced_sampling.R
\name{Spatially balanced sampling}
\alias{Spatially balanced sampling}
\alias{lpm_1}
\alias{lpm_1s}
\alias{lpm_2}
\alias{scps}
\alias{lcps}
\alias{lpm_2_hierarchical}
\title{Spatially balanced sampling}
\usage{
lpm_1(probabilities, spread_mat, ...)

lpm_1s(probabilities, spread_mat, ...)

lpm_2(probabilities, spread_mat, ...)

scps(probabilities, spread_mat, ...)

lcps(probabilities, spread_mat, ...)

lpm_2_hierarchical(probabilities, spread_mat, sizes, ...)
}
\arguments{
\item{probabilities}{A vector of inclusion probabilities.}

\item{spread_mat}{A matrix of spreading covariates.}

\item{...}{
  Arguments passed on to \code{\link[=.sampling_defaults]{.sampling_defaults}}
  \describe{
    \item{\code{eps}}{A small value used when comparing floats.}
    \item{\code{bucket_size}}{The maximum size of the k-d-tree nodes. A higher value gives a slower
k-d-tree, but is faster to create and takes up less memory.}
  }}

\item{sizes}{A vector of integers containing the sizes of the subsamples.}
}
\value{
A vector of sample indices, or in the case of hierarchical sampling, a matrix where the
first column contains sample indices and the second column contains subsample indices (groups).
}
\description{
Selects spatially balanced samples with prescribed inclusion probabilities from finite
populations.
}
\details{
\code{lpm_2_hierarchical} selects an initial sample using the LPM2 algorithm, and then splits this
sample into subsamples of given \code{sizes}, using successive, hierarchical selection with LPM2.
When using \code{lpm_2_hierarchical}, the inclusion probabilities must sum to an integer, and the
\code{sizes} vector (the subsamples) must sum to the same integer.
}
\section{Functions}{
\itemize{
\item \code{lpm_1()}: Local pivotal method 1

\item \code{lpm_1s()}: Local pivotal method 1s

\item \code{lpm_2()}: Local pivotal method 2

\item \code{scps()}: Spatially correlated Poisson sampling

\item \code{lcps()}: Locally correlated Poisson sampling

\item \code{lpm_2_hierarchical()}: Hierarchical Local pivotal method 2

}}
\examples{
set.seed(12345);
N = 1000;
n = 100;
prob = rep(n/N, N);
xs = matrix(runif(N * 2), ncol = 2);
sizes = c(10L, 20L, 30L, 40L);

s = lpm_1(prob, xs);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = lpm_1s(prob, xs);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = lpm_2(prob, xs);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = scps(prob, xs);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = lpm_2_hierarchical(prob, xs, sizes);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

\donttest{
s = lcps(prob, xs); # May have a long execution time
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

# Respects inclusion probabilities
set.seed(12345);
prob = c(0.2, 0.25, 0.35, 0.4, 0.5, 0.5, 0.55, 0.65, 0.7, 0.9);
N = length(prob);
xs = matrix(c(prob, runif(N * 2)), ncol = 3);

ep = rep(0L, N);
r = 10000L;

for (i in seq_len(r)) {
  s = lpm_2(prob, xs);
  ep[s] = ep[s] + 1L;
}

print(ep / r - prob);
}

}
\references{
Deville, J.-C., &  Tillé, Y. (1998).
Unequal probability sampling without replacement through a splitting method.
Biometrika 85, 89-101.

Grafström, A. (2012).
Spatially correlated Poisson sampling.
Journal of Statistical Planning and Inference, 142(1), 139-147.

Grafström, A., Lundström, N.L.P. & Schelin, L. (2012).
Spatially balanced sampling through the Pivotal method.
Biometrics 68(2), 514-520.

Lisic, J. J., & Cruze, N. B. (2016, June).
Local pivotal methods for large surveys.
In Proceedings of the Fifth International Conference on Establishment Surveys.

Prentius, W. (2024).
Locally correlated Poisson sampling.
Environmetrics, 35(2), e2832.
}
