% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_to_snirh.R
\name{convert_to_snirh}
\alias{convert_to_snirh}
\title{Convert data to SNIRH file format}
\usage{
convert_to_snirh(data, matrix, validate_stations = NULL, timeout = 30)
}
\arguments{
\item{data}{A data.frame or data.table containing the original laboratory data.
Must contain the following columns in order: snirh_entity, station_name,
station_id, sampling_date, parameter, unit, value.}

\item{matrix}{Character string specifying the type of matrix being processed.
Must be one of: "surface.water" or "biota".}

\item{validate_stations}{Logical. Whether to validate station IDs against
the SNIRH database. Defaults to TRUE for surface.water and biota matrices.
Requires internet connection.}

\item{timeout}{Numeric. Timeout in seconds for downloading station data.
Default is 30 seconds.}
}
\value{
A data.table formatted for SNIRH import with the following structure:
\itemize{
\item First row contains network specification (REDE=NETWORK_NAME)
\item Station identifiers (ESTACAO=STATION_ID) before each group of measurements
\item Date/time stamps in DD/MM/YYYY HH:MM format
\item Parameter values in SNIRH-compatible units and symbols
}
}
\description{
Cleans and converts laboratory data to the SNIRH (National
Information System on Water Resources) import format. It handles data
validation, unit conversions, station validation, and formatting according
to SNIRH standards.
}
\details{
The function performs several key operations:
\itemize{
\item Validates input data structure and removes empty rows/columns
\item Validates station IDs against SNIRH database (for surface.water and biota)
\item Checks for duplicate measurements (same station, date, and parameter)
\item Extracts pH temperature measurements when present
\item Converts measurement values to SNIRH-compatible units
\item Handles measurement flags (<, >, =) and special values
\item Formats output according to SNIRH import specifications
}
}
\section{Station Validation}{

For surface.water and biota matrices, the function validates that:
\itemize{
\item All station IDs exist in the SNIRH database
\item All stations have status "ATIVA" (active)
\item Internet connection is available for downloading station data
}

If validation fails, the function will stop and provide details about
invalid stations that need to be corrected in the database.
}

\section{Input Data Requirements}{

The input data must be a data.frame/data.table with exactly these columns:
\describe{
\item{snirh_entity}{Entity responsible for the data}
\item{station_name}{Human-readable station name}
\item{station_id}{Unique station identifier (must match SNIRH database)}
\item{sampling_date}{Date and time of sampling (POSIXct recommended)}
\item{parameter}{Parameter name as used in laboratory}
\item{unit}{Unit of measurement as used in laboratory}
\item{value}{Measured value (may include flags like <, >)}
}
}

\section{Parameter Conversion}{

Relies on an internal \code{parameters} dataset that maps laboratory
parameter names and units to SNIRH equivalents. This dataset must contain
conversion factors and SNIRH symbols for all parameters in the input data.
}

\examples{
# Example data structure
\donttest{
lab_data <- data.table::data.table(
  snirh_entity = "APA",
  station_name = "River station 1",
  station_id = "01F/01",  # Must be valid SNIRH station ID
  sampling_date = as.POSIXct("2024-01-15 10:30:00"),
  parameter = "pH - Campo",
  unit = "Escala Sorensen",
  value = "7.2"
)

# Convert surface water data (with station validation)
snirh_data <- convert_to_snirh(lab_data, "surface.water")

# Skip station validation if needed (not recommended)
snirh_data <- convert_to_snirh(lab_data, "surface.water",
                               validate_stations = FALSE)
}

}
