# valueprhr

Value-Price Analysis with Bayesian and Panel Data Methods

## Overview
This package provides tools for analyzing the relationship between direct 
prices (based on labor values) and prices of production using Bayesian 
generalized linear models, panel data methods, partial least squares 
regression, canonical correlation analysis, and panel vector autoregression.

## Installation

You can install the development version from GitHub:

```r
# install.packages("devtools")
devtools::install_github("isadorenabi/valueprhr")
```

## Main Functions

### Data Preparation
- `prepare_panel_data()`: Convert wide-format price matrices to panel data
- `prepare_log_matrices()`: Extract and log-transform price matrices
- `create_mundlak_data()`: Add Mundlak transformations to panel data

### Model Fitting
- `fit_bayesian_glm_sectors()`: Fit Bayesian GLM for each sector
- `fit_twoway_fe()`: Fit two-way fixed effects panel model
- `fit_mundlak_cre()`: Fit Mundlak correlated random effects model
- `fit_bayesian_hierarchical()`: Fit Bayesian mixed effects model
- `fit_pls_multivariate()`: Fit PLS regression with CV selection

### Multivariate Analysis
- `run_sparse_cca()`: Run sparse CCA with PCA preprocessing
- `fit_panel_var()`: Fit panel VAR model
- `fit_aggregated_var()`: Fit VAR on aggregated time series
- `panel_granger_test()`: Panel Granger causality tests

### Validation
- `rolling_window_cv()`: Rolling window cross-validation
- `leave_one_sector_out()`: LOSO cross-validation
- `summarize_cv_results()`: Summarize CV results

### Structural Break Tests
- `test_structural_breaks()`: Test for structural breaks
- `format_break_results()`: Format break test results
- `interpret_break_tests()`: Interpret break test results

### Summary and Comparison
- `compare_models()`: Generate model comparison table
- `generate_analysis_summary()`: Create comprehensive summary
- `run_full_analysis()`: Run complete analysis pipeline

## Example

```r
library(valueprhr)

# Create example data
set.seed(123)
years <- 2000:2019
sectors <- LETTERS[1:5]

direct <- data.frame(Year = years)
production <- data.frame(Year = years)

for (s in sectors) {
  direct[[s]] <- 100 + cumsum(rnorm(20, 2, 1))
  production[[s]] <- 102 + cumsum(rnorm(20, 2, 1))
}

# Run full analysis
results <- run_full_analysis(
  direct, production,
  run_bayesian = FALSE,
  run_cv = TRUE,
  run_breaks = TRUE
)

# View comparison table
print(results$comparison)
```

## Dependencies

### Required (Imports)
- stats
- utils
- Metrics

### Optional (Suggests)
- rstanarm: For Bayesian models
- loo: For LOO-CV
- plm: For panel data models
- pls: For PLS regression
- vars: For VAR models
- panelvar: For panel VAR
- strucchange: For structural break tests
- lmtest, sandwich: For robust standard errors

## Author

Jose Mauricio Gomez Julian (isadore.nabi@pm.me)

ORCID: 0009-0000-2412-3150

## License

MIT
