% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/precision_study.R
\name{precision_study}
\alias{precision_study}
\title{Precision Study Analysis}
\usage{
precision_study(
  data,
  value = "value",
  sample = NULL,
  site = NULL,
  day = "day",
  run = NULL,
  replicate = NULL,
  conf_level = 0.95,
  ci_method = c("satterthwaite", "mls", "bootstrap"),
  boot_n = 1999,
  method = c("anova", "reml")
)
}
\arguments{
\item{data}{A data frame containing the precision experiment data.}

\item{value}{Character string specifying the column name containing
measurement values. Default is \code{"value"}.}

\item{sample}{Character string specifying the column name for sample/level
identifier. Use when multiple concentration levels are tested. Default
is \code{NULL} (single sample).}

\item{site}{Character string specifying the column name for site/device
identifier. Use for multi-site reproducibility studies. Default is \code{NULL}
(single site).}

\item{day}{Character string specifying the column name for day identifier.
Default is \code{"day"}.}

\item{run}{Character string specifying the column name for run identifier
(within day). Default is \code{NULL} (assumes single run per day).}

\item{replicate}{Character string specifying the column name for replicate
identifier. If \code{NULL} (default), replicates are inferred from the data
structure.}

\item{conf_level}{Confidence level for intervals (default: 0.95).}

\item{ci_method}{Method for calculating confidence intervals:
\code{"satterthwaite"} (default) uses the Satterthwaite approximation,
\code{"mls"} uses the Modified Large Sample method,
\code{"bootstrap"} uses BCa bootstrap resampling.}

\item{boot_n}{Number of bootstrap resamples when \code{ci_method = "bootstrap"}
(default: 1999).}

\item{method}{Estimation method for variance components:
\code{"anova"} (default) uses ANOVA-based method of moments,
\code{"reml"} uses Restricted Maximum Likelihood (requires lme4 package).}
}
\value{
An object of class \code{c("precision_study", "valytics_precision", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with original data and metadata:
\itemize{
\item \code{data}: The input data frame (after validation)
\item \code{n}: Total number of observations
\item \code{n_excluded}: Number of observations excluded due to NAs
\item \code{factors}: Named list of factor column names used
\item \code{value_col}: Name of the value column
}
}
\item{design}{List describing the experimental design:
\itemize{
\item \code{type}: Design type (e.g., "single_site", "multi_site")
\item \code{structure}: Character string describing nesting (e.g., "day/run")
\item \code{levels}: Named list with number of levels for each factor
\item \code{balanced}: Logical; TRUE if design is balanced
\item \code{n_samples}: Number of distinct samples/concentration levels
}
}
\item{variance_components}{Data frame with variance component estimates:
\itemize{
\item \code{component}: Name of variance component
\item \code{variance}: Estimated variance
\item \code{sd}: Standard deviation (sqrt of variance
\item \code{pct_total}: Percentage of total variance
\item \code{df}: Degrees of freedom
}
}
\item{precision}{Data frame with precision estimates:
\itemize{
\item \code{measure}: Precision measure name (repeatability, intermediate, etc.)
\item \code{sd}: Standard deviation
\item \code{cv_pct}: Coefficient of variation (percent)
\item \code{ci_lower}: Lower confidence limit
\item \code{ci_upper}: Upper confidence limit
}
}
\item{anova_table}{ANOVA table with SS, MS, DF for each source of variation}
\item{by_sample}{If multiple samples: list of results per sample}
\item{settings}{List with analysis settings}
\item{call}{The matched function call}
}
}
\description{
Performs variance component analysis for precision experiments following
established methodology for clinical laboratory method validation. Estimates
repeatability, Within-laboratory precision, and reproducibility from nested
experimental designs.
}
\details{
This function implements variance component analysis for nested experimental
designs commonly used in clinical laboratory precision studies. The analysis
follows methodology consistent with international standards.

\strong{Supported Experimental Designs:}

\itemize{
\item \strong{Single-site, day/run/replicate}: Classic 20 x 2 x 2 design
(20 days, 2 runs per day, 2 replicates per run)
\item \strong{Single-site, day/replicate}: Simplified design without run factor
(e.g., 5 days x 5 replicates for verification)
\item \strong{Multi-site}: 3 sites x 5 days x 5 replicates for reproducibility
\item \strong{Custom designs}: Any fully-nested combination of factors
}

\strong{Variance Components:}

For a design with site/day/run/replicate, the model is:
\deqn{y_{ijkl} = \mu + S_i + D_{j(i)} + R_{k(ij)} + \epsilon_{l(ijk)}}

where S = site, D = day (nested in site), R = run (nested in day),
and epsilon = residual error.

\strong{Precision Measures:}

\itemize{
\item \strong{Repeatability}: Within-run precision (sqrt of error variance)
\item \strong{Between-run precision}: Additional variability between runs
\item \strong{Between-day precision}: Additional variability between days
\item \strong{Within-laboratory precision}: Within-laboratory precision
(combines day, run, and error variance)
\item \strong{Reproducibility}: Total precision including between-site
variability (for multi-site designs)
}
}
\section{Confidence Intervals}{

Three methods are available for confidence interval estimation:

\itemize{
\item \strong{Satterthwaite} (default): Uses Satterthwaite's approximation
for degrees of freedom of linear combinations of variance components.
\item \strong{MLS}: Modified Large Sample method, which can provide better
coverage when variance components may be estimated as negative.
\item \strong{Bootstrap}: BCa bootstrap resampling. Most robust but
computationally intensive.
}
}

\section{ANOVA vs REML}{

\itemize{
\item \strong{ANOVA} (default): Method of moments estimation. Works well for
balanced designs. May produce negative variance estimates for small
variance components (set to zero by default).
\item \strong{REML}: Restricted Maximum Likelihood. Preferred for unbalanced
designs. Requires the lme4 package. Always produces non-negative
estimates.
}
}

\examples{
# Example with simulated precision data
set.seed(42)

# Generate study design: 20 days x 2 runs x 2 replicates
n_days <- 20
n_runs <- 2
n_reps <- 2

prec_data <- expand.grid(
  day = 1:n_days,
  run = 1:n_runs,
  replicate = 1:n_reps
)

# Add realistic variance components
day_effect <- rep(rnorm(n_days, 0, 1.5), each = n_runs * n_reps)
run_effect <- rep(rnorm(n_days * n_runs, 0, 1.0), each = n_reps)
error <- rnorm(nrow(prec_data), 0, 2.0)

prec_data$value <- 100 + day_effect + run_effect + error

# Run precision study
prec <- precision_study(
  data = prec_data,
  value = "value",
  day = "day",
  run = "run"
)

print(prec)
summary(prec)

}
\references{
Chesher D (2008). Evaluating assay precision. \emph{Clinical Biochemist
Reviews}, 29(Suppl 1):S23-S26.

ISO 5725-2:2019. Accuracy (trueness and precision) of measurement methods
and results - Part 2: Basic method for the determination of repeatability
and reproducibility of a standard measurement method.

Searle SR, Casella G, McCulloch CE (1992). \emph{Variance Components}.
Wiley, New York.

Satterthwaite FE (1946). An approximate distribution of estimates of
variance components. \emph{Biometrics Bulletin}, 2:110-114.
}
\seealso{
\code{\link[=verify_precision]{verify_precision()}} for comparing results to manufacturer claims,
\code{\link[=plot.precision_study]{plot.precision_study()}} for visualization,
\code{\link[=summary.precision_study]{summary.precision_study()}} for detailed summary
}
